<?php
/**
 * This file is part of D3mocracy.
 * 
 * D3mocracy is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * at your option) any later version.

 * D3mocracy is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with D3mocracy.  If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Saša T. <sasa@mindframes.org>
 */

define("IN_MYBB", 1);
define('THIS_SCRIPT', 'd3mocracy.php');
$templatelist =
	'd3mocracy_avatar_add,'.
	'd3mocracy_avatar_add_script,'.
	'd3mocracy_avatar_box,'.
	'd3mocracy_avatar_box_hidden_container,'.
	'd3mocracy_avatar_box_new,'.
	'd3mocracy_delete_button,'.
	'd3mocracy_header,'.
	'd3mocracy_page,'.
	'd3mocracy_window';

require_once "./global.php";
require_once MYBB_ROOT."inc/functions_image.php";
require_once MYBB_ROOT."inc/functions_upload.php";
require_once MYBB_ROOT."inc/plugins/d3mocracy/config.php";
require_once MYBB_ROOT."inc/plugins/d3mocracy/functions.php";
d3mocracyLoadClasses();
$lang->load("d3mocracy");

// In case D3mocracy is deactivated/not installed
if(!isset($templates->cache['d3mocracy_header']))
	die($lang->d3mocracy_disabled);

$printer = new D3mocracyOutputPrinter($lang, $templates, $charset, $stylesheets);
$dbAvatars = new D3mocracyAvatarsDB($db);
$dbUsers = new D3mocracyUsersDB($db);
$dbVotes = new D3mocracyVotesDB($db);

if($mybb->input['action'] === 'window' && isset($mybb->input['ownerid']))
{
	$ownerID = (int)$mybb->input['ownerid'];
	$window = new D3mocracyWindow();
	$windowView = new D3mocracyWindowView($window);
	$windowActions = new D3mocracyWindowActions($window);
	$requester = new D3mocracyRequester(null, $mybb);
	
	// Owner ID invalid
	if(!$ownerID)
		$printer->jsonError($lang->d3mocracy_invalid_user, 'invalid_user');
	
	$owner = $dbUsers->fetchOwner($ownerID, $window);
	
	// Owner not found
	if(!$owner)
		$printer->jsonError($lang->d3mocracy_invalid_user, 'invalid_user');
	
	// Owner disabled d3mocracy
	if($owner->status == 'off')
		$printer->jsonError($lang->d3mocracy_user_disabled, 'user_disabled');
	
	// Let Window object choose which avatars are to be displayed
	$window->pickAvatars($dbAvatars->fetchAvatars($requester, $owner->uid, null));
	$windowActions->handleExpiredAvatars($owner, $dbAvatars, $dbVotes, $dbUsers, $updated);
	
	// If profile avatar is not on the list of avatars, add it
	if(!empty($owner->avatar) && !$owner->isProfileAvatarD3mocratic() && !$window->getNumAvatars())
	{
		$avatar = $owner->getOwnersAvatar();
		$avatarActions = new D3mocracyAvatarActions($avatar);
		$avatarActions->moveAvatar($mybb, $dbUsers);
		$dbAvatars->insertAvatar($avatar);
		$dbUsers->recalcAvatarCount($ownerID);
		$window->addAvatar($avatar);
		$owner->setAvatarParams($avatar);
	}
	else if($owner->avatar !== $window->getActiveAvatar()->url)
	{
		$activeAvatar = $window->getActiveAvatar();
		$dbUsers->setProfileAvatar($owner->uid, $activeAvatar);
		if($owner->avatartype == 'upload' && strpos($owner->avatar, $mybb->settings['avataruploadpath']) === 0)
			@unlink(MYBB_ROOT.'/'.d3mocracyGetAvatarFileName($owner->avatar, false));
		$owner->setAvatarParams($activeAvatar);
	}
	
	// Create and associate avatars with View Components
	foreach($window->getAvatars() as $avatar)
	{
		$avatarView = new D3mocracyAvatarView($avatar, $requester, $mybb, $lang);
		$windowView->addAvatarView($avatarView);
	}
	
	// Add avatarNew box
	$avatarNewView = new D3mocracyAvatarNewView($requester, $owner);
	$windowView->addAvatarNewView($avatarNewView);
	
	// No boxes to display
	if(!$windowView->getNumBoxes())
		$printer->jsonError($lang->d3mocracy_user_no_avatars, 'user_no_avatars');
	
	$windowView->recalculate();
	
	// Print window
	$printer->html($windowView->render($templates, $owner, $lang));
}

else if($mybb->input['action'] === 'deleteavatar' && $mybb->request_method == "post" && isset($mybb->input['avatarid']))
{
	$deleteAvatarId = (int)$mybb->input['avatarid'];
	$statuses = $dbUsers->fetchRequesterOwnerStatus($mybb->user['uid'], $deleteAvatarId);
	list($requesterD3Status, $ownerD3Status) = $statuses;
	
	$requester = new D3mocracyRequester($requesterD3Status, $mybb);
	$window = new D3mocracyWindow();
	$windowActions = new D3mocracyWindowActions($window);
	
	// If requester cannot act, send error
	if(!$requester->canAct())
		$printer->jsonError($lang->d3mocracy_no_perms_delete, 'no_perms');
	
	// Owner disabled d3mocracy
	if($ownerD3Status === 'off')
		$printer->jsonError($lang->d3mocracy_user_disabled, 'user_disabled');
	
	// If CSRF token is invalid, send error
	if(generate_post_check() !== $mybb->input['csrftoken'])
		$printer->jsonError($lang->d3mocracy_invalid_post_key, 'invalid_post_key');
	
	// If invalid avatar ID, send error
	if($deleteAvatarId < 1)
		$printer->jsonError($lang->d3mocracy_invalid_avatar, 'invalid_avatar');
	
	$avatars = $dbAvatars->fetchOwnerAvatarsFromAvatarID($deleteAvatarId);
	
	// No avatars, exit
	if(!count($avatars))
		$printer->jsonError($lang->d3mocracy_invalid_avatar, 'invalid_avatar');
	
	// Pick and handle expired avatars
	$owner = new D3mocracyOwner(array('uid' => $avatars[0]->ownerid), $window);
	$window->pickAvatars($avatars);
	$windowActions->handleExpiredAvatars($owner, $dbAvatars, $dbVotes, $dbUsers, $updated);
	$avatarToDelete = $window->getAvatarByID($deleteAvatarId);
	
	// Avatar to delete was not found, exit with error
	if($avatarToDelete === null)
		$printer->jsonError($lang->d3mocracy_expired_avatar, 'expired_avatar');
	
	$avatarToDeleteWasActive = $avatarToDelete->status == 'active';
	
	// Delete avatar if requester has permission
	if($requester->canDeleteAvatar($avatarToDelete))
	{
		$dbAvatars->deleteAvatarsVotes(array($avatarToDelete));
		$dbUsers->recalcAvatarCount($avatarToDelete->ownerid);
		$window->removeAvatar($avatarToDelete);
	}
	// Error and exit otherwise
	else
		$printer->jsonError($lang->d3mocracy_no_perms_delete, 'no_perms');
	
	$avatarsJson = array();
	
	// If the deleted avatar was active, need to set new active avatar	
	if($avatarToDeleteWasActive)
	{
		// There are inactive avatars, first one become active
		$newActiveAvatar = $window->getActiveAvatar();
		if($newActiveAvatar instanceof D3mocracyAvatar)
		{
			$newActiveAvatar->activateInactive();
			$newActiveAvatarView = new D3mocracyAvatarView($newActiveAvatar, $requester, $mybb, $lang);
			
			$avatarsJson[] = $newActiveAvatarView->getDateJsonParams();
			$dbAvatars->updateActiveAvatar($newActiveAvatar->avatarid);
		}
		// There are no inactive avatars, set empty avatar parameters
		else
			$dbUsers->setProfileAvatar($owner->uid);
	}
	
	// Successfully exit
	$printer->json(array('deleted' => true, 'avatars' => $avatarsJson));
}

else if($mybb->input['action'] === 'voteavatar' && $mybb->request_method == "post" && isset($mybb->input['avatarid']) && isset($mybb->input['avatarvote']))
{
	$vote = new D3mocracyVote(array('avatarid' => $mybb->input['avatarid'],
		'voterid' => $mybb->user['uid'],'type' => $mybb->input['avatarvote']));
	
	$statuses = $dbUsers->fetchRequesterOwnerStatus($mybb->user['uid'], $vote->avatarid);
	list($requesterD3Status, $ownerD3Status) = $statuses;
	
	$requester = new D3mocracyRequester($requesterD3Status, $mybb);
	$window = new D3mocracyWindow();
	$windowActions = new D3mocracyWindowActions($window);
	
	// If requester cannot act, send error
	if(!$requester->canAct())
		$printer->jsonError($lang->d3mocracy_no_perms_vote, 'no_perms');
	
	// If requester isn't a d3mocrat, send error
	if($requester->status === 'off')
		$printer->jsonError($lang->d3mocracy_actions_suspended, 'no_perms');
	
	// Owner disabled d3mocracy
	if($ownerD3Status === 'off')
		$printer->jsonError($lang->d3mocracy_user_disabled, 'user_disabled');
	
	// If CSRF token is invalid, send error
	if(generate_post_check() !== $mybb->input['csrftoken'])
		$printer->jsonError($lang->d3mocracy_invalid_post_key, 'invalid_post_key');
	
	// If invalid avatar ID, send error
	if($vote->avatarid < 1)
		$printer->jsonError($lang->d3mocracy_invalid_avatar, 'invalid_avatar');
	
	// If invalid vote type, send error
	if(!in_array($vote->type, array('like', 'dislike')))
		$printer->jsonError($lang->d3mocracy_invalid_vote_type, 'invalid_vote_type');
	
	$avatars = $dbAvatars->fetchAvatars($requester, null, $vote->avatarid);
	
	// No avatars, exit
	if(!count($avatars))
		$printer->jsonError($lang->d3mocracy_invalid_avatar, 'invalid_avatar');
	
	// Pick and handle expired avatars
	$owner = new D3mocracyOwner(array('uid' => $avatars[0]->ownerid), $window);
	$window->pickAvatars($avatars);
	$windowActions->handleExpiredAvatars($owner, $dbAvatars, $dbVotes, $dbUsers, $updated);
	
	$avatarToVote = $window->getAvatarByID($vote->avatarid);
	
	// If avatar to vote doesn't exist, send error
	if(!$avatarToVote)
		$printer->jsonError($lang->d3mocracy_expired_avatar, 'expired_avatar');
	
	$avatarToVoteView = new D3mocracyAvatarView($avatarToVote, $requester, $mybb, $lang);
	
	// Getting pre and post active avatars, appending vote locally and to database
	$preActiveAvatar = $window->getActiveAvatar();
	$voteDiff = $avatarToVote->appendVote($vote);
	$dbVotes->vote($vote, $voteDiff->mode);
	$window->sortAvatars();
	$postActiveAvatar = $window->getActiveAvatar();
	$avatarsJson = array();
	
	// Voted avatar needs to be activated
	if($preActiveAvatar !== $avatarToVote && $postActiveAvatar === $avatarToVote)
	{
		$dbAvatars->updateVoteCacheAndStatus($voteDiff, 'active', $avatarToVote->status);
		$dbAvatars->updateInactiveAvatar($avatarToVote->ownerid, $avatarToVote->avatarid);
		
		// Get Json date params
		$avatarToVote->activateInactive();
		$avatarsJson[] = $avatarToVoteView->getDateJsonParams();
		
		$preActiveAvatar->deactivateActive();
		$preActiveAvatarView = new D3mocracyAvatarView($preActiveAvatar, $requester, $mybb, $lang);
		$avatarsJson[] = $preActiveAvatarView->getDateJsonParams();
	}
	
	// Voted avatar needs to be deactivated
	else if($preActiveAvatar === $avatarToVote && $postActiveAvatar !== $avatarToVote)
	{
		$dbAvatars->updateVoteCacheAndStatus($voteDiff, 'inactive', $avatarToVote->status);
		$dbAvatars->updateActiveAvatar($postActiveAvatar->avatarid);
		
		// Get Json date params
		$avatarToVote->deactivateActive();
		$avatarsJson[] = $avatarToVoteView->getDateJsonParams();
		
		$postActiveAvatar->activateInactive();
		$postActiveAvatarView = new D3mocracyAvatarView($postActiveAvatar, $requester, $mybb, $lang);
		$avatarsJson[] = $postActiveAvatarView->getDateJsonParams();
	}
	
	// No activation necessary
	else
		$dbAvatars->updateVoteCacheAndStatus($voteDiff, $avatarToVote->status, $avatarToVote->status);
	
	$json = array
	(
		'action' => $voteDiff->mode,
		'vote' => $vote->type,
		'likes' => $avatarToVote->likes,
		'dislikes' => $avatarToVote->dislikes,
		'classRating' => $avatarToVoteView->getRatingClass(),
		'classLk' => $avatarToVoteView->getLikeClass(),
		'classDk' => $avatarToVoteView->getDislikeClass(),
		'titleLk' => $avatarToVoteView->getLikeTitle(),
		'titleDk' => $avatarToVoteView->getDislikeTitle(),
		'avatars' => $avatarsJson
	);
	
	// Successfully exit
	$printer->json($json);
}

else if($mybb->input['action'] === 'addavatar' && isset($mybb->input['ownerid']))
{
	$lang->load("usercp");
	$avatarError = '';
	$ownerID = (int)$mybb->input['ownerid'];
	$requesterD3Status = $dbUsers->fetchUserStatus($mybb->user['uid']);
	$requester = new D3mocracyRequester($requesterD3Status, $mybb);
	$window = new D3mocracyWindow();
	$windowActions = new D3mocracyWindowActions($window);
	
	// If requester cannot act, exit with error
	if(!$requester->canAct())
		$printer->htmlError($lang->d3mocracy_error, $lang->d3mocracy_no_perms_add);
	
	// If requester isn't a d3mocrat, send error
	if($requester->status === 'off')
		$printer->htmlError($lang->d3mocracy_error, $lang->d3mocracy_actions_suspended);
	
	// If invalid owner ID, exit with error
	if(!$ownerID)
		$printer->htmlError($lang->d3mocracy_error, $lang->d3mocracy_invalid_user);
	
	$owner = $dbUsers->fetchOwnerRequesterVote($ownerID, $requester->getID(), $window);
	
	// If owner doesn't exist, exit with error
	if(!$owner)
		$printer->htmlError($lang->d3mocracy_error, $lang->d3mocracy_invalid_user);
	
	// If owner doesn't accept avatar submissions, exit with error
	if($owner->status === 'off')
		$printer->htmlError($lang->d3mocracy_error, $lang->d3mocracy_user_disabled);
	
	// If requester already submitted an avatar for this user, exit with error
	if($owner->requestersubmitted)
		$printer->htmlError($lang->d3mocracy_error, $lang->d3mocracy_already_uploaded);
	
	// If avatar cap is full, exit with error
	if($owner->getNumAvatars() >= D3MOCRACY_MAX_AVATARS_PER_USER)
		$printer->htmlError($lang->d3mocracy_error, $lang->d3mocracy_avatar_cap_full);
	
	// If requester can't submit an avatar
	if(!$requester->canSubmitAvatar($owner))
		$printer->htmlError($lang->d3mocracy_error, $lang->d3mocracy_no_perms_add);
	
	$mybb->settings['avataruploadpath'] = d3mocracyGetUploadDirName();
	$submitFormView = new D3mocracySubmitFormView($owner, $lang);
	
	// Submitting avatar
	if($mybb->request_method == "post")
	{
		if(generate_post_check() !== $mybb->input['csrftoken'])
			$printer->htmlError($lang->d3mocracy_error, $lang->d3mocracy_invalid_post_key);
		
		// Submit avatar, exit if exception is thrown
		try
		{
			list($updatedAvatar, $avatarError) = mybbDoAvatar();
		}
		catch(Exception $e)
		{
			$printer->htmlError($lang->d3mocracy_error, $e->getMessage());
		}
		
		// Submission failed, render page with avatar error message
		if(!empty($avatarError))
			$printer->html($submitFormView->render($templates, $mybb, $theme, $charset, $stylesheets, $avatarError));
		
		// Moving avatar
		if($updatedAvatar["avatartype"] == 'upload')
		{
			$avatarFilename = d3mocracyGetAvatarFileName($updatedAvatar['avatar']);
			$newAvatarFilename = d3mocracyFilenameReplaceStr($avatarFilename, 'avatar_'.$owner->uid.'_by_'.$requester->getID());
			@rename($mybb->settings['avataruploadpath'].'/'.$avatarFilename, $mybb->settings['avataruploadpath'].'/'.$newAvatarFilename);
			$updatedAvatar['avatar'] = $mybb->settings['avataruploadpath'].'/'.$newAvatarFilename.'?dateline='.TIME_NOW;
		}
		
		// If moving of the file failed, delete file and exit with error
		if(!file_exists($mybb->settings['avataruploadpath'].'/'.$newAvatarFilename))
		{
			@unlink($mybb->settings['avataruploadpath'].'/'.$avatarFilename);
			$printer->htmlError($lang->d3mocracy_error, $lang->d3mocracy_error_uploading_avatar);
		}
		
		$submittedView = new D3mocracySubmittedView($owner, $lang);
		$submittedAvatar = D3mocracyAvatar::createFromParams($owner, $requester, $updatedAvatar);
		$submittedAvatarView = new D3mocracyAvatarView($submittedAvatar, $requester, $mybb, $lang);
		
		$windowView = new D3mocracyWindowView($window);
		$windowView->addAvatarView($submittedAvatarView);
		
		// Fetching other avatars to see if reactivation is necessary
		$window->pickAvatars($dbAvatars->fetchOwnerAvatarsFromOwnerID($owner->uid));
		$windowActions->handleExpiredAvatars($owner, $dbAvatars, $dbVotes, $dbUsers, $updated);
		
		// Getting pre-active avatar to deactivate later if necessary
		$preActiveAvatar = $window->getActiveAvatar();
		if($window->getNumAvatars() > 0)
			$preActiveAvatar->setStatusActive();
		
		// Finally, add newly submitted and sort to determine if it activates
		$window->addAvatar($submittedAvatar);
		$window->sortAvatars();
		$avatarsJson = array();

		// Insert and recalculate after fetch and before reactivation
		$dbAvatars->insertAvatar($submittedAvatar);
		$dbUsers->recalcAvatarCount($owner->uid);
		
		// Newly added avatar needs activation
		if($window->getActiveAvatar() === $submittedAvatar)
		{
			$submittedAvatar->setStatusActive();
			$dbAvatars->updateActiveAvatar($submittedAvatar->avatarid);
			$dbAvatars->updateInactiveAvatar($owner->uid, $submittedAvatar->avatarid);
			$owner->setAvatarParams($submittedAvatar);
			
			// Get Json date params
			if(isset($preActiveAvatar) && $preActiveAvatar instanceof D3mocracyAvatar)
			{
				$preActiveAvatar->deactivateActive();
				$preActiveAvatarView = new D3mocracyAvatarView($preActiveAvatar, $requester, $mybb, $lang);
				$avatarsJson[] = $preActiveAvatarView->getDateJsonParams();
			}
		}
		$windowView->recalculate();
		
		// Render page
		$printer->html($submittedView->render($submittedAvatarView, $windowView, $avatarsJson, $templates, $mybb, $charset, $stylesheets));
	}
	// Display avatar submission form
	else
	{
		// Render page with no errors
		$printer->html($submitFormView->render($templates, $mybb, $theme, $charset, $stylesheets, ''));
	}
}