<?php
/**
 * This file is part of D3mocracy.
 * 
 * D3mocracy is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * at your option) any later version.

 * D3mocracy is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with D3mocracy.  If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Saša T. <sasa@mindframes.org>
 */

/**
 * Avatar Model
 */
class D3mocracyAvatar extends D3mocracyModel
{
	/**
	 * ID of the avatar
	 *
	 * @var int
	 */
	public $avatarid;
	
	/**
	 * Owner's user ID
	 *
	 * @var int
	 */
	public $ownerid;
	
	/**
	 * Submitter's user ID
	 *
	 * @var int
	 */
	public $submitterid;
	
	/**
	 * Avatar type (remote/upload)
	 *
	 * @var string
	 */
	public $type;
	
	/**
	 * Avatar dimensions (i.e. 100|80)
	 *
	 * @var string
	 */
	public $dimensions;
	
	/**
	 * Avatar URL
	 *
	 * @var string
	 */
	public $url;
	
	/**
	 * Like count
	 *
	 * @var int
	 */
	public $likes;
	
	/**
	 * Dislike count
	 *
	 * @var int
	 */
	public $dislikes;
	
	/**
	 * Status (active/inactive)
	 *
	 * @var string
	 */
	public $status;
	
	/**
	 * Time of submission (timestamp)
	 *
	 * @var int
	 */
	public $submitted;
	
	/**
	 * Time of status change (timestamp)
	 *
	 * @var int
	 */
	public $since;
	
	/**
	 * Requester's vote (like/dislike/NULL)
	 *
	 * @var string
	 */
	public $requestervote;
	
	/**
	 * Submitter's username
	 *
	 * @var string
	 */
	public $submittername;
	
	/**
	 * Returns this model's parameter names
	 *
	 * @return array Array with parameter names as keys and types as values
	 */
	public function getParamNames()
	{
		return array
		(
			'avatarid' => 'int',
			'ownerid' => 'int',
			'submitterid' => 'int',
			'type' => 'string',
			'dimensions' => 'string',
			'url' => 'string',
			'likes' => 'int',
			'dislikes' => 'int',
			'status' => 'string',
			'submitted' => 'int',
			'since' => 'int',
			'requestervote' => 'mixed',
			'submittername' => 'string',
		);
	}
	
	/**
	 * Appends vote to cache and returns Vote Diff Model with likes/dislikes difference
	 *
	 * @param D3mocracyVote $vote Vote Model instance of the vote to append
	 * @return D3mocracyAvatarVoteDiff Vote Diff Model instance with likes/dislikes difference, null if avatarids don't match
	 */
	public function appendVote(D3mocracyVote $vote)
	{
		if($vote->avatarid != $this->avatarid)
			return null;
		
		$voteDiff = array('avatarid' => $vote->avatarid);
		
		// Mode added
		if(!$this->requestervote)
		{
			$voteDiff['pluslikes']    = $vote->type === 'like'    ? 1 : 0;
			$voteDiff['plusdislikes'] = $vote->type === 'dislike' ? 1 : 0;
			$voteDiff['mode'] = 'added';
			$this->requestervote = $vote->type;
		}
		// Mode updated
		else if($this->requestervote !== $vote->type)
		{
			$voteDiff['pluslikes']    = $vote->type === 'like'    ? 1 : -1;
			$voteDiff['plusdislikes'] = $vote->type === 'dislike' ? 1 : -1;
			$voteDiff['mode'] = 'updated';
			$this->requestervote = $vote->type;
		}
		// Mode removed
		else if($this->requestervote === $vote->type)
		{
			$voteDiff['pluslikes']    = $vote->type === 'like'    ? -1 : 0;
			$voteDiff['plusdislikes'] = $vote->type === 'dislike' ? -1 : 0;
			$voteDiff['mode'] = 'removed';
			$this->requestervote = null;
		}
		
		// Appending vote locally
		$this->likes += $voteDiff['pluslikes'];
		$this->dislikes += $voteDiff['plusdislikes'];
		
		return new D3mocracyAvatarVoteDiff($voteDiff);
	}
	
	/**
	 * Creates new instance from Owner Model, Requester Model and avatar parameters
	 *
	 * @param D3mocracyOwner $owner Owner Model instance
	 * @param D3mocracyRequester $requester Requester Model instance; Requester acts as submitter
	 * @param array $avatarParams Avatar parameters containing 'avatar', 'avatartype' and 'avatardimensions'
	 * @return D3mocracyAvatar New Avatar Model instace
	 */
	public static function createFromParams(D3mocracyOwner $owner, D3mocracyRequester $requester, $avatarParams)
	{
		$avatar = array
		(
			'avatarid' => '',
			'ownerid' => $owner->uid,
			'submitterid' => $requester->getID(),
			'submittername' => $requester->getUsername(),
			'type' => $avatarParams['avatartype'],
			'dimensions' => $avatarParams['avatardimensions'],
			'url' => $avatarParams['avatar'],
			'likes' => '0',
			'dislikes' => '0',
			'status' => 'inactive',
			'submitted' => TIME_NOW,
			'since' => TIME_NOW,
		);
		return new D3mocracyAvatar($avatar);
	}
	
	/**
	 * Returns avatar rating
	 *
	 * @return int Avatar rating
	 */
	public function getRating()
	{
		return $this->likes - $this->dislikes;
	}
	
	/**
	 * Returns avatar's width
	 *
	 * @return int Avatar's width
	 */
	public function getWidth()
	{
		$dimensions = explode('|', $this->dimensions);
		return (int)$dimensions[0];
	}
	
	/**
	 * Returns avatar's height
	 *
	 * @return int Avatar's height
	 */
	public function getHeight()
	{
		$dimensions = explode('|', $this->dimensions);
		return (int)$dimensions[1];
	}
	
	/**
	 * Deactivates this avatar if status is active
	 */
	public function deactivateActive()
	{
		if($this->isActiveAndExpired())
		{
			$this->status = 'inactive';
			$this->requestervote = null;
			$this->likes = $this->dislikes = 0;
			$this->since += D3MOCRACY_ACTIVITY_TIME_SPAN;
		}
		else
		{
			$this->status = 'inactive';
			$this->since = TIME_NOW;
		}
	}
	
	/**
	 * Activates this avatar if status is inactive
	 */
	public function activateInactive()
	{
		$this->status = 'active';
		$this->since = TIME_NOW;
	}
	
	/**
	 * Sets status to active
	 */
	public function setStatusActive()
	{
		$this->status = 'active';
	}
	
	/**
	 * Sets status to inactive
	 */
	public function setStatusInactive()
	{
		$this->status = 'inactive';
	}
	
	/**
	 * Set status to active and bring since parameter up to date
	 *
	 * @param int $since Starting since parameter
	 */
	public function loopActivate($since = null)
	{
		$this->setStatusActive();
		if(is_int($since))
			$this->since = $since;
		
		$loops = 0;
		while($this->isActiveAndExpired())
		{
			// Deactivate and activate again, updating the since parameter
			$this->deactivateActive();
			$this->setStatusActive();
			if(($loops ++) > 10000) break;
		}
	}
	
	/**
	 * Returns whether this avatar is inactive and has expired
	 *
	 * @return bool True if it's expired, false if it hasn't
	 */
	public function isInactiveAndExpired()
	{
		return $this->status == 'inactive' && (TIME_NOW - $this->since) > D3MOCRACY_INACTIVITY_TIME_SPAN;
	}
	
	/**
	 * Returns whether this avatar is active and has expired
	 *
	 * @return bool True if it's expired, false if it hasn't
	 */
	public function isActiveAndExpired()
	{
		return $this->status == 'active' && (TIME_NOW - $this->since) > D3MOCRACY_ACTIVITY_TIME_SPAN;
	}
}