<?php
/**
 * This file is part of D3mocracy.
 * 
 * D3mocracy is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * at your option) any later version.

 * D3mocracy is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with D3mocracy.  If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Saša T. <sasa@mindframes.org>
 */

/**
 * Avatar box View Component
 */
class D3mocracyAvatarView
{
	/**
	 * Instance of D3mocracyAvatar Model
	 *
	 * @var D3mocracyAvatar
	 */
	private $avatar;
	
	/**
	 * Instance of D3mocracyRequester Model
	 *
	 * @var D3mocracyRequester
	 */
	private $requester;
	
	/**
	 * Instance of MyLanguage MyBB class
	 *
	 * @var MyLanguage
	 */
	private $lang;
	
	/**
	 * Maximum avatar width
	 *
	 * @var int
	 */
	private $maxWidth;
	
	/**
	 * Maximum avatar height
	 *
	 * @var int
	 */
	private $maxHeight;
	
	/**
	 * This avatar's scaled display width
	 *
	 * @var int
	 */
	private $viewWidth;
	
	/**
	 * This avatar's scaled display height
	 *
	 * @var int
	 */
	private $wiewHeight;
	
	/**
	 * Constructor; Sets model instances, MyBB object and language object
	 *
	 * @param D3mocracyAvatar $avatar Avatar Model instance
	 * @param D3mocracyRequester $requester Requester Model instance
	 * @param MyBB $mybb MyBB object
	 * @param MyLanguage $lang MyBB language object
	 */
	public function __construct
	(
		D3mocracyAvatar $avatar,
		D3mocracyRequester $requester,
		MyBB $mybb,
		MyLanguage $lang
	){
		$this->avatar = $avatar;
		$this->requester = $requester;
		$this->mybb = $mybb;
		$this->lang = $lang;
		$this->setMaxDimensions($mybb->settings['postmaxavatarsize']);
	}
	
	/**
	 * Sets individual maximum avatar dimensions from string
	 *
	 * @param string $maxDimensions Dimensions in the form 80x80 or 80|80
	 */
	public function setMaxDimensions($maxDimensions)
	{
		if($maxDimensions)
		{
			$maxDimensions = str_ireplace('x', '|', $maxDimensions);
			list($this->maxWidth, $this->maxHeight) = explode('|', $maxDimensions);
			$this->maxWidth = (int)$this->maxWidth;
			$this->maxHeight = (int)$this->maxHeight;
		}
	}
	
	/**
	 * Calculates avatar's display width and/or returns it
	 *
	 * @return string Display width
	 */
	public function getViewWidth()
	{
		if($this->viewWidth)
		{
			// Return stored width
			return $this->viewWidth;
		}
		if(!$this->maxWidth)
		{
			// Return natural width
			return $this->avatar->getWidth();
		}
		else
		{
			// Calculate, store and return
			$scaled = $this->getScaledDimensions();
			$this->viewWidth = $scaled['width'];
			$this->viewHeight = $scaled['height'];
			return $scaled['width'];
		}
	}
	
	/**
	 * Calculates avatar's display height and/or returns it
	 *
	 * @return string Display height
	 */
	public function getViewHeight()
	{
		if($this->viewHeight)
		{
			// Return stored height
			return $this->viewHeight;
		}
		if(!$this->maxHeight)
		{
			// Return natural height
			return $this->avatar->getHeight();
		}
		else
		{
			// Calculate, store and return
			$scaled = $this->getScaledDimensions();
			$this->viewWidth = $scaled['width'];
			$this->viewHeight = $scaled['height'];
			return $scaled['height'];
		}
	}
	
	/**
	 * Returns scaled dimensions based on maximum dimensions maintaining proportions
	 *
	 * @return array Array with scaled dimensions in the form array('width' => .., 'height' => ..)
	 */
	public function getScaledDimensions()
	{
		return scale_image
		(
			$this->avatar->getWidth(),
			$this->avatar->getHeight(),
			$this->maxWidth,
			$this->maxHeight
		);
	}
	
	/**
	 * Returns rating class indicating positive, negative or neutral rating
	 *
	 * @return string Rating class name
	 */
	public function getRatingClass()
	{
		$rating = $this->avatar->getRating();
		
		if($rating  > 0)
			return 'd3mocracy_rating_positive';
		if($rating  < 0)
			return 'd3mocracy_rating_negative';
		if($rating == 0)
			return 'd3mocracy_rating_neutral';
	}
	
	/**
	 * Returns like button class indicating whether requester has liked avatar
	 *
	 * @return string Like button class name
	 */
	public function getLikeClass()
	{
		return $this->avatar->requestervote === 'like' ?
			'd3mocracy_voted_like' : 'd3mocracy_vote_like';
	}
	
	/**
	 * Returns dislike button class indicating whether requester has disliked avatar
	 *
	 * @return string Dislike button class name
	 */
	public function getDislikeClass()
	{
		return $this->avatar->requestervote === 'dislike' ?
			'd3mocracy_voted_dislike' : 'd3mocracy_vote_dislike';
	}
	
	/**
	 * Returns like button title describing requester's potential request
	 *
	 * @return string Like button title
	 */
	public function getLikeTitle()
	{
		return $this->avatar->requestervote === 'like' ?
			$this->lang->d3mocracy_remove_like : $this->lang->d3mocracy_like_avatar;
	}
	
	/**
	 * Returns dislike button title describing requester's potential request
	 *
	 * @return string Dislike button title
	 */
	public function getDislikeTitle()
	{
		return $this->avatar->requestervote === 'dislike' ?
			$this->lang->d3mocracy_remove_dislike : $this->lang->d3mocracy_dislike_avatar;
	}
	
	/**
	 * Returns class name for the avatar box indicating whether requester has submitted the avatar
	 *
	 * @return string Class name if requester is submitter, empty string otherwise
	 */
	public function getSubmittedByMeClass()
	{
		return $this->avatar->submitterid == $this->requester->getID() ?
			' d3mocracy_submitted_by_me' : '';
	}
	
	/**
	 * Returns display formatted text and date indicating time of submission
	 *
	 * @return string Formatted text and date
	 */
	public function getSubmittedTip()
	{
		return $this->lang->d3mocracy_submitted.' '.
			my_date(D3MOCRACY_DATE_FORMAT, $this->avatar->submitted).', '.
			my_date(D3MOCRACY_TIME_FORMAT, $this->avatar->submitted);
	}
	
	/**
	 * Returns display formatted text and date indicating status change time
	 *
	 * @return string Formatted text and date
	 */
	public function getSinceTip()
	{
		$langEntry = $this->avatar->status == 'active' ?
			$this->lang->d3mocracy_active_since : $this->lang->d3mocracy_inactive_since;
		
		return $langEntry.' '.
			my_date(D3MOCRACY_DATE_FORMAT, $this->avatar->since).', '.
			my_date(D3MOCRACY_TIME_FORMAT, $this->avatar->since);
	}
	
	/**
	 * Returns display formatted text and date indicating expiration time
	 *
	 * @return string Formatted text and date
	 */
	public function getExpiresTip()
	{
		$expTime = $this->avatar->since + ($this->avatar->status == 'active' ?
			D3MOCRACY_ACTIVITY_TIME_SPAN : D3MOCRACY_INACTIVITY_TIME_SPAN);
		
		return $this->lang->d3mocracy_expires.' '.
			my_date(D3MOCRACY_DATE_FORMAT, $expTime).', '.
			my_date(D3MOCRACY_TIME_FORMAT, $expTime);
	}
	
	/**
	 * Returns date params to be used in JSON response
	 *
	 * @return array Json date parameters
	 */
	public function getDateJsonParams()
	{
		return array
		(
			'avatarid' => $this->avatar->avatarid,
			'since' => $this->getSinceTip(),
			'expires' => $this->getExpiresTip()
		);
	}
	
	/**
	 * Renders avatar box's template and returns output
	 *
	 * @param templates $templates Instance of templates, MyBB templating object
	 * @param D3mocracyWindowView $windowView Instance of WindowView object
	 * @return string Rendered avatar box's HTML
	 */
	public function render(templates $templates, D3mocracyWindowView $windowView)
	{
		// Variables for easy template access
		$avatar = $this->avatar;
		$lang = $this->lang;
		$avatarView = $this;
		$deleteButton = '';
		
		// Get output
		if($this->requester->canDeleteAvatar($this->avatar))
			eval('$deleteButton = "'.$templates->get("d3mocracy_delete_button").'";');
		eval('$avatarBox = "'.$templates->get("d3mocracy_avatar_box").'";');
		
		// Return output
		return $avatarBox;
	}
}