<?php
/**
 * This file is part of D3mocracy.
 * 
 * D3mocracy is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * at your option) any later version.

 * D3mocracy is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with D3mocracy.  If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Saša T. <sasa@mindframes.org>
 */

/**
 * Avatars DB class
 */
class D3mocracyAvatarsDB
{
	/**
	 * Database object
	 *
	 * @var mixed
	 */
	private $db;
	
	/**
	 * Constructor; Sets datebase object
	 */
	public function __construct($db)
	{
		$this->db = $db;
	}
	
	/**
	 * Fetches owner's avatars and some requester info
	 *
	 * @param D3mocracyRequester $requester Requester Model instance
	 * @param int $ownerID ID of the owner whose avatars to fetch
	 * @param int $avatarID ID of the avatar used to determine ID of the owner
	 * @return array Recordset with owner's avatars or empty array if owner has no avatars
	 */
	public function fetchAvatars(D3mocracyRequester $requester, $ownerID, $avatarID)
	{
		// Owner ID is known
		if(is_int($ownerID) && $ownerID > 0)
			$ownerPart = '"'.$this->db->escape_string($ownerID).'"';
		
		// Owner ID is not known, determine through subquery
		else if(is_int($avatarID) && $avatarID > 0)
			$ownerPart = '(SELECT `ownerid`
				FROM `'.TABLE_PREFIX.'d3mocracy_avatars`
				WHERE `avatarid` = "'.$this->db->escape_string($avatarID).'")';
		
		$query = $this->db->query('SELECT `a`.`avatarid`, `a`.`ownerid`, `a`.`submitterid`, `u`.`username` AS `submittername`, `a`.`type`,
				`a`.`dimensions`, `a`.`url`, `a`.`likes`, `a`.`dislikes`, `a`.`likes` - `a`.`dislikes` AS `rating`,
				'.($requester->canAct() ? '`v`.`type` AS `requestervote`, ' : 'NULL AS `requestervote`, ').'
				`a`.`status`, `a`.`submitted`, `a`.`since`
			FROM `'.TABLE_PREFIX.'d3mocracy_avatars` `a`
			LEFT JOIN `'.TABLE_PREFIX.'users` `u`
			ON `a`.`submitterid` = `u`.`uid`
			'.($requester->canAct() ? 'LEFT JOIN `'.TABLE_PREFIX.'d3mocracy_votes` `v`
			ON `v`.`voterid` = "'.$this->db->escape_string($requester->getID()).'"
			AND `v`.`avatarid` = `a`.`avatarid` ' : '').'
			WHERE `a`.`ownerid` = '.$ownerPart.'
			GROUP BY `a`.`avatarid`
			ORDER BY `rating` DESC, `submitted` ASC');
		
		if($this->db->num_rows($query) > 0)
		{
			while($avatar = $this->db->fetch_array($query))
				$avatars[] = new D3mocracyAvatar($avatar);
			return $avatars;
		}
		else
			return array();
	}

	/**
	 * Fetches owner's avatars from avatarid pointing to ownerid
	 *
	 * @param int $avatarid Avatar ID to use in subquery for determining owner ID
	 * @return array Recordset with owner's avatars or empty array if owner has no avatars
	 */
	public function fetchOwnerAvatarsFromAvatarID($avatarid)
	{
		$query = $this->db->query('SELECT *
			FROM `'.TABLE_PREFIX.'d3mocracy_avatars`
			WHERE `ownerid` = (SELECT `ownerid`
				FROM `'.TABLE_PREFIX.'d3mocracy_avatars`
				WHERE `avatarid` = "'.$this->db->escape_string($avatarid).'")
			ORDER BY (`likes` - `dislikes`) DESC, `submitted` ASC');
		
		if($this->db->num_rows($query) > 0)
		{
			while($avatar = $this->db->fetch_array($query))
				$avatars[] = new D3mocracyAvatar($avatar);
			return $avatars;
		}
		else
			return array();
	}
	
	/**
	 * Fetches owner's avatars from Owner ID
	 *
	 * @param int $ownerID ID of the owner whose avatars to fetch
	 * @return array Recordset with owner's avatars or empty array if owner has no avatars
	 */
	public function fetchOwnerAvatarsFromOwnerID($ownerID)
	{
		$query = $this->db->query('SELECT *
			FROM `'.TABLE_PREFIX.'d3mocracy_avatars`
			WHERE `ownerid` = "'.$this->db->escape_string($ownerID).'"
			ORDER BY (`likes` - `dislikes`) DESC, `submitted` ASC');
		
		if($this->db->num_rows($query) > 0)
		{
			while($avatar = $this->db->fetch_array($query))
				$avatars[] = new D3mocracyAvatar($avatar);
			return $avatars;
		}
		else
			return array();
	}
	
	/**
	 * Fetches only unordered avatarid, type and url, needed for deletetion
	 *
	 * @param int $ownerID ID of the owner whose avatars to fetch
	 * @return array Recordset with owner's avatars or empty array if owner has no avatars
	 */
	public function fetchAvatarsForDeletion($ownerID)
	{
		$query = $this->db->query('SELECT `avatarid`, `type`, `url`
			FROM `'.TABLE_PREFIX.'d3mocracy_avatars`
			WHERE `ownerid` = "'.$this->db->escape_string($ownerID).'"');
		
		if($this->db->num_rows($query) > 0)
		{
			while($avatar = $this->db->fetch_array($query))
				$avatars[] = new D3mocracyAvatar($avatar);
			return $avatars;
		}
		else
			return array();
	}
	
	/**
	 * Fetches avatar count of the user identified by ownerID or total avatar count
	 *
	 * @param int $ownerID Avatars' owner ID or null
	 * @return int Avatar count
	 */
	public function fetchNumAvatars($ownerID = null)
	{
		$query = $this->db->query('SELECT COUNT(*) AS `count`
			FROM `'.TABLE_PREFIX.'d3mocracy_avatars`'.
			(is_int($ownerID) ? 'WHERE `ownerid` = "'.$this->db->escape_string($ownerID).'"' : ''));
		
		return (int)$this->db->fetch_field($query, 'count');
	}
	
	/**
	 * Inserts avatar into database
	 *
	 * @param D3mocracyAvatar $avatar Avatar to delete (Model instance)
	 * @return int Avatar's newly inserted ID
	 */
	function insertAvatar(D3mocracyAvatar $avatar)
	{
		$this->db->write_query('INSERT INTO `'.TABLE_PREFIX.'d3mocracy_avatars`
			(`ownerid`,`submitterid`,`type`,`dimensions`,`url`,`likes`,`dislikes`,`status`,`submitted`,`since`)
			VALUES
			(
				"'.$this->db->escape_string($avatar->ownerid).'",
				"'.$this->db->escape_string($avatar->submitterid).'",
				"'.$this->db->escape_string($avatar->type).'",
				"'.$this->db->escape_string($avatar->dimensions).'",
				"'.$this->db->escape_string($avatar->url).'",
				"'.$this->db->escape_string($avatar->likes).'",
				"'.$this->db->escape_string($avatar->dislikes).'",
				"'.$this->db->escape_string($avatar->status).'",
				"'.$this->db->escape_string($avatar->submitted).'",
				"'.$this->db->escape_string($avatar->since).'"
			)');
		$insertedID = (int)$this->db->insert_id();
		$avatar->avatarid = (int)$insertedID;
		return $insertedID;
	}
	
	/**
	 * Deletes avatars from database and disk and their corresponding votes
	 *
	 * @param array $avatars Avatars to delete (D3mocracyAvatar instances)
	 */
	public function deleteAvatarsVotes(array $avatars)
	{
		$avatarIDs = array();
		$avatarUrls = array();
		
		// Loop through avatars and extract urls and IDs
		foreach($avatars as $key => $avatar)
		{
			// Add avatar ID
			$avatarIDs[$key] = $this->db->escape_string($avatar->avatarid);
			
			// Remove query from url and add it
			if($avatar->type === 'upload')
				$avatarUrls[$key] = substr($avatar->url, 0, strrpos($avatar->url, '?'));
		}
		if(count($avatarIDs) > 0)
		{
			// Deleting from database
			$this->db->write_query('DELETE `a`, `v`
				FROM `'.TABLE_PREFIX.'d3mocracy_avatars` `a`
				LEFT JOIN `'.TABLE_PREFIX.'d3mocracy_votes` `v`
				ON `a`.`avatarid` = `v`.`avatarid`
				WHERE `a`.`avatarid` IN ("'.join('","', $avatarIDs).'")');
				
			// Deleting from disk
			foreach($avatarUrls as $avatarUrl)
				@unlink(MYBB_ROOT.$avatarUrl);
		}
	}
	
	/**
	 * Activates avatar and sets it on owner's profile
	 *
	 * @param int $newActiveAvatarID ID of the avatar to activate
	 * @param int $since Since parameter to update
	 */
	public function updateActiveAvatar($newActiveAvatarID, $since = TIME_NOW)
	{
		$this->db->write_query('UPDATE `'.TABLE_PREFIX.'users` `u`
			LEFT JOIN `'.TABLE_PREFIX.'d3mocracy_avatars` `a`
			ON `u`.`uid` = `a`.`ownerid`
			SET `a`.`status` = "active", `a`.`since` = "'.$this->db->escape_string($since).'", `u`.`avatar` = `a`.`url`, `u`.`avatardimensions` = `a`.`dimensions`, `u`.`avatartype` = `a`.`type`
			WHERE `a`.`avatarid` = "'.$this->db->escape_string($newActiveAvatarID).'"');
	}
	
	/**
	 * Activates avatar but doesn't set it as profile avatar
	 *
	 * @param int $newActiveAvatarID ID of the avatar to activate
	 * @param int $since Since parameter to update
	 */
	public function updateActiveAvatarOnly($newActiveAvatarID, $since = TIME_NOW)
	{
		$this->db->write_query('UPDATE `'.TABLE_PREFIX.'d3mocracy_avatars`
			SET `status` = "active", `since` = "'.$this->db->escape_string($since).'"
			WHERE `avatarid` = "'.$this->db->escape_string($newActiveAvatarID).'"');
	}
	
	/**
	 * Deactivates avatars identified by ownerID and NOT by activeAvatarID
	 *
	 * @param int $ownerID Owner ID whose avatar is to be deleted
	 * @param int $activeAvatarID Avatar ID which is NOT to be deleted
	 */
	public function updateInactiveAvatar($ownerID, $activeAvatarID)
	{
		$this->db->write_query('UPDATE `'.TABLE_PREFIX.'d3mocracy_avatars`
			SET `status` = "inactive", `since` = "'.TIME_NOW.'"
			WHERE `status` = "active"
				AND `ownerid` = "'.$this->db->escape_string($ownerID).'"
				AND `avatarid` != "'.$this->db->escape_string($activeAvatarID).'"');
	}
	
	/**
	 * Updates status and since, sets likes and dislikes to zero
	 *
	 * @param D3mocracyAvatar $avatar D3mocracyAvatar instance of the avatar to deactivate
	 */
	public function reactivateResetVoteCache(D3mocracyAvatar $avatar)
	{
		$this->db->write_query('UPDATE `'.TABLE_PREFIX.'d3mocracy_avatars`
			SET `status` = "'.$this->db->escape_string($avatar->status).'",
				`since` = "'.$this->db->escape_string($avatar->since).'",
				`likes` = "0", `dislikes` = "0"
			WHERE `avatarid` = "'.$this->db->escape_string($avatar->avatarid).'"');
	}
	
	/**
	 * Updates vote like/dislike cache in avatars table and sets new status/since
	 *
	 * @param D3mocracyAvatarVoteDiff $voteDiff Vote Diff Model instance
	 * @param string $newStatus New status to set
	 * @param string $oldStatus Old status to use as reference
	 */
	public function updateVoteCacheAndStatus($voteDiff, $newStatus, $oldStatus)
	{
		$selection = '';
		$likes = $voteDiff->pluslikes;
		$dislikes = $voteDiff->plusdislikes;
		
		// Generating the dynamic part of the query
		foreach(array('likes', 'dislikes') as $type)
			$selection .= $$type !== 0 ? ("`a`.`{$type}` = `a`.`{$type}` ".($$type >= 0 ? '+ '.$$type : '- '.substr($$type, 1)).', ') : '';
		
		if($newStatus == 'active' || $newStatus == 'inactive')
		{
			$selection .= '`a`.`status` = "'.$newStatus.'", ';
			
			// Since parameter is updated only when status is changed from active to inactive and vise-versa
			if(($newStatus == 'active' && $oldStatus == 'inactive') || ($oldStatus == 'active' && $newStatus == 'inactive'))
				$selection .= '`a`.`since` = "'.TIME_NOW.'", ';
			
			// Profile avatar is updated only if avatar is activated
			if($newStatus == 'active' && $oldStatus == 'inactive')
				$selection .= '`u`.`avatar` = `a`.`url`, `u`.`avatardimensions` = `a`.`dimensions`, `u`.`avatartype` = `a`.`type`, ';
		}
		
		$selection = substr($selection, 0, -2);
		$this->db->write_query('UPDATE '.
			($newStatus == 'active' ? '`'.TABLE_PREFIX.'users` `u` LEFT JOIN ' : '').
			'`'.TABLE_PREFIX.'d3mocracy_avatars` `a` '.
			($newStatus == 'active' ? 'ON `u`.`uid` = `a`.`ownerid` ' : '').
			'SET '.$selection.' WHERE `a`.`avatarid` = "'.$this->db->escape_string($voteDiff->avatarid).'"');
	}
	
	/**
	 * Recounts avatars for users starting at given position through given quantity
	 *
	 * @param int $position Start position for the LIMIT clause
	 * @param int $perPage Quantity for the LIMIT clause
	 */
	public function recountAvatars($position, $perPage)
	{
		$query = $this->db->query('SELECT `userid`
			FROM `'.TABLE_PREFIX.'d3mocracy_users`
			ORDER BY `userid`
			LIMIT '."{$position}, {$perPage}");
		
		if($this->db->num_rows($query) > 0)
		{
			$userids = '';
			while($userid = $this->db->fetch_field($query, 'userid'))
				$userids .= $userid.',';
			$userids = substr($userids, 0, -1);
			
			$this->db->write_query('
				UPDATE `'.TABLE_PREFIX.'d3mocracy_users` `u`
				LEFT JOIN (SELECT `ownerid`, COUNT(*) AS `count`
					FROM `'.TABLE_PREFIX.'d3mocracy_avatars`
					WHERE `ownerid` IN ('.$userids.')
					GROUP BY `ownerid`) `a`
				ON `u`.`userid` = `a`.`ownerid`
				SET `u`.`avatarcount` = IFNULL(`a`.`count`, 0)
				WHERE `u`.`userid` IN ('.$userids.')');
		}
	}
}