<?php
/**
 * This file is part of D3mocracy.
 * 
 * D3mocracy is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * at your option) any later version.

 * D3mocracy is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with D3mocracy.  If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Saša T. <sasa@mindframes.org>
 */

/**
 * Owner Model
 */
class D3mocracyOwner extends D3mocracyModel
{
	/**
	 * Owner's user ID
	 *
	 * @var int
	 */
	public $uid;
	
	/**
	 * Owner's username
	 *
	 * @var string
	 */
	public $username;
	
	/**
	 * Avatar URL
	 *
	 * @var string
	 */
	public $avatar;
	
	/**
	 * Avatar type (remote/upload)
	 *
	 * @var string
	 */
	public $avatartype;
	
	/**
	 * Avatar dimensions (i.e. 80|80)
	 *
	 * @var string
	 */
	public $avatardimensions;
	
	/**
	 * D3mocracy status (on/off)
	 *
	 * @var string
	 */
	public $status;
	
	/**
	 * Avatar count
	 *
	 * @var int
	 */
	public $avatarcount;
	
	/**
	 * Whether requester has submitted avatar for the owner
	 *
	 * @var bool
	 */
	public $requestersubmitted;
	
	/**
	 * D3mocracyWindow instance
	 *
	 * @var D3mocracyWindow
	 */
	private $window;
	
	/**
	 * Returns this model's parameter names
	 *
	 * @return array Array with parameter names as keys and types as values
	 */
	public function getParamNames()
	{
		return array
		(
			'uid' => 'int',
			'username' => 'string',
			'avatar' => 'string',
			'avatartype' => 'string',
			'avatardimensions' => 'string',
			'status' => 'string',
			'avatarcount' => 'int',
			'requestersubmitted' => 'bool',
		);
	}
	
	/**
	 * Constructor; Sets params and Window Model
	 *
	 * @param array $params Model parameters to set
	 * @param D3mocracyWindow $window Window Model to set
	 */
	public function __construct(array $params, D3mocracyWindow $window)
	{
		parent::__construct($params);
		$this->window = $window;
	}
	
	/**
	 * Creates and returns D3mocracyOwner instance using parameters from D3mocracyRequester instance
	 *
	 * @param D3mocracyRequester $requester Requester to turn into Owner
	 * @param D3mocracyWindow $window Window Model instance to pass
	 */
	public static function createFromRequester(D3mocracyRequester $requester, D3mocracyWindow $window)
	{
		$params = array
		(
			'uid' => $requester->getID(),
			'username' => $requester->getUsername(),
		);
		return new D3mocracyOwner($params, $window);
	}
	
	/**
 	 * Sets avatar parameters from Avatar Model instance
 	 *
 	 * @param D3mocracyAvatar $avatar Avatar Model instance or null
 	 */
	public function setAvatarParams($avatar = null)
	{
		if($avatar instanceof D3mocracyAvatar)
		{
			$this->avatar = $avatar->url;
			$this->avatartype = $avatar->type;
			$this->avatardimensions = $avatar->dimensions;
		}
		else
		{
			$this->avatar = '';
			$this->avatartype = '';
			$this->avatardimensions = '';
		}
	}
	
	/**
	 * Returns owner's avatar count
	 *
	 * @return int Avatar count
	 */
	public function getNumAvatars()
	{
		if($this->avatarcount !== null)
			return $this->avatarcount;
		else
			return $this->window->getNumAvatars();
	}
	
	/**
	 * Returns avatar's width
	 *
	 * @return int Avatar's width
	 */
	public function getAvatarWidth()
	{
		$dimensions = explode('|', $this->avatardimensions);
		return (int)$dimensions[0];
	}
	
	/**
	 * Returns avatar's height
	 *
	 * @return int Avatar's height
	 */
	public function getAvatarHeight()
	{
		$dimensions = explode('|', $this->avatardimensions);
		return (int)$dimensions[1];
	}
	
	/**
	 * Returns avatar (Model instance) submitted by his owner with default values
	 *
	 * @return D3mocracyAvatar Avatar Model instance
	 */
	public function getOwnersAvatar()
	{
		return new D3mocracyAvatar
		(
			array
			(
				'avatarid' => '',
				'ownerid' => $this->uid,
				'submitterid' => $this->uid,
				'submittername' => $this->username,
				'type' => $this->avatartype,
				'dimensions' => $this->avatardimensions,
				'url' => $this->avatar,
				'likes' => '0',
				'dislikes' => '0',
				'requestervote' => null,
				'status' => 'active',
				'submitted' => TIME_NOW,
				'since' => TIME_NOW,
			)
		);
	}
	
	/**
	 * Returns whether profile avatar exists among the ones submitted through D3mocracy
	 *
	 * @return bool True if it's D3mocracic, false if it's Non-d3mocratic
	 */
	public function isProfileAvatarD3mocratic()
	{
		foreach($this->window->getAvatars() as $avatar)
		{
			if($avatar->url === $this->avatar)
			{
				return true;
			}
		}
		return false;
	}
	
	/**
	 * Returns whether owner has received current requester's avatar
	 *
	 * @param D3mocracyRequester $requester Requester Model instance
	 * @return bool True if owner has requester's avatar, false if he doesn't
	 */
	public function hasRequestersAvatar(D3mocracyRequester $requester)
	{
		foreach($this->window->getAvatars() as $avatar)
		{
			if($avatar->submitterid == $requester->getID())
			{
				return true;
			}
		}
		return false;
	}
}