<?php
/**
 * This file is part of D3mocracy.
 * 
 * D3mocracy is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * at your option) any later version.

 * D3mocracy is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with D3mocracy.  If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Saša T. <sasa@mindframes.org>
 */

/**
 * Requester Model
 */
class D3mocracyRequester extends D3mocracyModel
{
	/**
	 * D3mocracy status
	 *
	 * @var string
	 */
	public $status;
	
	/**
	 * MyBB instance
	 *
	 * @var MyBB
	 */
	private $mybb;
	
	/**
	 * Returns this model's parameter names
	 *
	 * @return array Array with parameter names as keys and types as values
	 */
	public function getParamNames()
	{
		return array('status' => 'mixed');
	}
	
	/**
	 * Constructor; Sets parameters and MyBB instance
	 *
	 * @param string $status Requester's D3mocracy status
	 * @param MyBB $mybb Instance of MyBB
	 */
	public function __construct($status, MyBB $mybb)
	{
		parent::__construct(array('status' => $status));
		$this->mybb = $mybb;
	}
	
	/**
	 * Returns whether requester can generally act
	 *
	 * @return bool True if he can act, false if he cannot
	 */
	public function canAct()
	{
		// Requester can act if:
		return
			// Must not be in the banned group
			(int)$this->mybb->usergroup['isbannedgroup'] != 1 &&
			// Must not be a guest
			(int)$this->mybb->usergroup['gid'] != 1 &&
			// Must not be awaiting activation
			(int)$this->mybb->usergroup['gid'] != 5 &&
			// Must not be banned
			(int)$this->mybb->usergroup['gid'] != 7 &&
			// Must be able to view threads
			(int)$this->mybb->usergroup['canviewthreads'];
	}
	
	/**
	 * Returns whether requester can submit an avatar for the owner
	 *
	 * @param D3mocracyOwner $owner Owner whom requester is submitting avatar
	 * @return bool True if he can submit, false if he cannot
	 */
	public function canSubmitAvatar(D3mocracyOwner $owner)
	{
		// Requester can submit an avatar if the owner hasn't disabled d3mocracy,
		// requester hasn't already submitted one and avatar cap is not full
		return
			$this->canAct() &&
			$owner->status !== 'off' &&
			!$owner->hasRequestersAvatar($this) &&
			$owner->getNumAvatars() < D3MOCRACY_MAX_AVATARS_PER_USER;
	}
	
	/**
	 * Returns whether requester can delete the supplied avatar
	 *
	 * @param D3mocracyAvatar $avatar Avatar which requester wants to delete
	 * @return bool True if he can delete it, false if he cannot
	 */
	public function canDeleteAvatar(D3mocracyAvatar $avatar)
	{
		$canAct = $this->canAct();
		
		// Requester can delete avatar if:
		return
			// ...submitter is the same as requester
			($canAct && $avatar->submitterid == $this->getID()) ||
			// ...requester is admin or supermod
			($this->isSuperMod());
	}
	
	/**
	 * Returns whether requester is Super Moderator
	 *
	 * @return bool True if he is, false if he isn't
	 */
	public function isSuperMod()
	{
		return $this->mybb->usergroup['issupermod'] == '1';
	}
	
	/**
	 * Returns requester's user ID
	 *
	 * @return int Requester's user ID
	 */
	public function getID()
	{
		return (int)$this->mybb->user['uid'];
	}
	
	/**
	 * Returns requester's username
	 *
	 * @return string Requester's username
	 */
	public function getUsername()
	{
		return $this->mybb->user['username'];
	}
}