<?php
/**
 * This file is part of D3mocracy.
 * 
 * D3mocracy is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * at your option) any later version.

 * D3mocracy is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with D3mocracy.  If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Saša T. <sasa@mindframes.org>
 */

/**
 * Users DB class
 */
class D3mocracyUsersDB
{
	/**
	 * Database object
	 *
	 * @var mixed
	 */
	private $db;
	
	/**
	 * Constructor; Sets datebase object
	 */
	public function __construct($db)
	{
		$this->db = $db;
	}
	
	/**
	 * Fetches owner parameters
	 *
	 * @param int $ownerID Owner's user ID
	 * @param D3mocracyWindow $window Window Model instance to pass to Owner Model
	 * @return array Owner parameters or null if owner was not found
	 */
	public function fetchOwner($ownerID, $window)
	{
		$query = $this->db->query('
			SELECT `u`.`uid`, `u`.`username`, `u`.`avatar`, `u`.`avatardimensions`, `u`.`avatartype`, `du`.`status`
			FROM `'.TABLE_PREFIX.'d3mocracy_users` `du`
			RIGHT JOIN `'.TABLE_PREFIX.'users` `u`
			ON `du`.`userid` = `u`.`uid`
			WHERE `u`.`uid` = '.$this->db->escape_string($ownerID).'
			LIMIT 1');
		
		if($this->db->num_rows($query) > 0)
			return new D3mocracyOwner($this->db->fetch_array($query), $window);
		else
			return null;
	}
	
	/**
	 * Fetches owner parameters along with requester's vote
	 *
	 * @param int $ownerID Owner's user ID
	 * @param int $requesterID Requester's user ID used to determine requester's vote
	 * @param D3mocracyWindow $window Window Model instance to pass to Owner Model
	 * @return array Owner parameters or null if owner was not found
	 */
	public function fetchOwnerRequesterVote($ownerID, $requesterID, $window)
	{
		$query = $this->db->query('
			SELECT `u`.`uid`, `u`.`username`, `du`.`status`, `du`.`avatarcount`, `a`.`avatarid` IS NOT NULL AS `requestersubmitted`
			FROM `'.TABLE_PREFIX.'users` `u`
			LEFT JOIN `'.TABLE_PREFIX.'d3mocracy_users` `du`
			ON `du`.`userid` = `u`.`uid`
			LEFT JOIN `'.TABLE_PREFIX.'d3mocracy_avatars` `a`
			ON `u`.`uid` = `a`.`ownerid` AND `a`.`submitterid` = "'.$this->db->escape_string($requesterID).'"
			WHERE `u`.`uid` = "'.$this->db->escape_string($ownerID).'"
			LIMIT 1');
		
		if($this->db->num_rows($query) > 0)
			return new D3mocracyOwner($this->db->fetch_array($query), $window);
		else
			return null;
	}
	
	/**
	 * Fetches all users who posted in the supplied thread ID
	 *
	 * @param int $tid Thread ID of the thread from which to fetch users
	 * @return array Array containing users' parameters
	 */
	public function fetchUsersFromTid($tid)
	{
		$query = $this->db->query('SELECT `p`.`uid`, `u`.`avatar` = "" AS `emptyavatar`, `du`.`status`, `du`.`avatarcount`
			FROM `'.TABLE_PREFIX.'posts` `p`
			LEFT JOIN `'.TABLE_PREFIX.'users` `u`
			ON `p`.`uid` = `u`.`uid`
			LEFT JOIN `'.TABLE_PREFIX.'d3mocracy_users` `du`
			ON `p`.`uid` = `du`.`userid`
			WHERE `p`.`tid` = "'.$this->db->escape_string($tid).'"
			GROUP BY `p`.`uid`');
		
		if($this->db->num_rows($query) > 0)
		{
			while($user = $this->db->fetch_array($query))
				$users[] = $user;
			return $users;
		}
		else
			return array();
	}
	
	/**
	 * Fetches whether user exists in the D3mocracy users table
	 *
	 * @param int $userID User ID
	 * @return bool True is the user exists, false if he doesn't
	 */
	public function fetchUserExists($userID)
	{
		// Getting user count to determine whether to insert or update
		$query = $this->db->query('SELECT COUNT(*) AS `count`
			FROM `'.TABLE_PREFIX.'d3mocracy_users`
			WHERE `userid` = "'.$userID.'"');
		$count = $this->db->fetch_array($query);
		return (int)$count['count'] > 0;
	}
	
	/**
	 * Fetches user's D3mocracy status
	 *
	 * @param int $userID User ID
	 * @return string 'off' or 'on'
	 */
	public function fetchUserStatus($userID)
	{
		if((int)$userID < 1)
			return 'off';
		
		$query = $this->db->query('SELECT `status`
			FROM `'.TABLE_PREFIX.'d3mocracy_users`
			WHERE `userid` = "'.$this->db->escape_string($userID).'"');
		
		if(!$this->db->num_rows($query))
			return 'on';
		
		$status = $this->db->fetch_array($query);
		return $status['status'] === 'off' ? 'off' : 'on';
	}
	
	/**
	 * Fetches requester's D3mocracy status based on requester ID and owner's D3mocracy status based on owner's Avatar ID
	 *
	 * @param int $requesterID Requester's user ID
	 * @param int $ownerAvatarID Avatar ID to use in determining owner's user ID
	 * @return array Array with the following elements: 0/'requester' => {requester status}, 1/'owner' => {owner status}
	 */
	public function fetchRequesterOwnerStatus($requesterID, $ownerAvatarID)
	{
		$query = $this->db->query('
		(SELECT "requester" AS `user`, `status`
			FROM `'.TABLE_PREFIX.'d3mocracy_users`
			WHERE `userid` = "'.$this->db->escape_string($requesterID).'")
		UNION
		(SELECT "owner" AS `user`, `status`
			FROM `'.TABLE_PREFIX.'d3mocracy_users`
			WHERE `userid` = (SELECT `ownerid`
				FROM `'.TABLE_PREFIX.'d3mocracy_avatars`
				WHERE `avatarid` = "'.$this->db->escape_string($ownerAvatarID).'"))');
		
		$final[0] = $final[1] = $final['requester'] = $final['owner'] = 'on';
		while($user = $this->db->fetch_array($query))
		{
			switch($user['user'])
			{
				case 'requester':
					$final[0] = $final['requester'] = $user['status'];
					break;
				
				case 'owner':
					$final[1] = $final['owner'] = $user['status'];
					break;
			}
		}
		return $final;
	}
	
	/**
	 * Fetches total number of users
	 *
	 * @return int Total number of users
	 */
	public function fetchNumUsers()
	{
		$query = $this->db->query('SELECT COUNT(*) AS `count`
			FROM `'.TABLE_PREFIX.'d3mocracy_users`');
		return (int)$this->db->fetch_field($query, 'count');
	}
	
	/**
	 * Updates user's D3mocracy status
	 *
	 * @param int $userID User ID
	 * @param string $status 'off' or 'on'
	 */
	public function updateUserStatus($userID, $status)
	{
		$status = $status === 'off' ? 'off' : 'on';
		if($this->fetchUserExists($userID))
		{
			$this->db->write_query('UPDATE `'.TABLE_PREFIX.'d3mocracy_users`
				SET `status` = "'.$status.'"
				WHERE `userid` = "'.$userID.'"');
		}
		else
			$this->recalcAvatarCount($userID, $status, 'insert');
	}
	
	/**
	 * Inserts or updates recalculated avatar count into database
	 *
	 * @param int $ownerID Owner ID whose avatar count to recalculate
	 * @param string $status D3mocracy status
	 * @param string $mode 'insert' or 'update'
	 */
	function recalcAvatarCount($ownerID, $status = 'on', $mode = null)
	{
		$ownerID = $this->db->escape_string($ownerID);
		
		// Subquery to for getting avatar count
		$subquery = '(SELECT COUNT(*)
			FROM `'.TABLE_PREFIX.'d3mocracy_avatars`
			WHERE `ownerid` = "'.$ownerID.'")';
		
		// Update query, if owner is present
		$updateQuery = 'UPDATE `'.TABLE_PREFIX.'d3mocracy_users`
			SET `avatarcount` = '.$subquery.'
			WHERE `userid` = "'.$ownerID.'"';
		
		// Insert query, if owner is not present
		$insertQuery = 'INSERT INTO `'.TABLE_PREFIX.'d3mocracy_users`
			(`userid`,`status`,`avatarcount`)
			VALUES ("'.$ownerID.'", "'.$status.'", '.$subquery.')';
		
		// Mode specified directly
		if($mode == 'update')
			$this->db->write_query($updateQuery);
		else if($mode == 'insert')
			$this->db->write_query($insertQuery);
		
		// Detect mode
		if($this->fetchUserExists($ownerID))
			$this->db->write_query($updateQuery);
		else
			$this->db->write_query($insertQuery);
	}
	
	/**
	 * Deletes user from D3mocracy list
	 *
	 * @param int ID of the user to delete
	 */
	public function deleteUser($userID)
	{
		$this->db->write_query('DELETE
			FROM `'.TABLE_PREFIX.'d3mocracy_users`
			WHERE `userid` = "'.$this->db->escape_string($userID).'"');
	}
	
	/**
	 * Sets profile avatar to the user specified by uid
	 *
	 * @param int $uid User ID to set parameters to
	 * @param D3mocracyAvatar $avatar Avatar Model instance holding the parameters or null
	 */
	public function setProfileAvatar($uid, $avatar = null)
	{
		if($avatar instanceof D3mocracyAvatar)
		{
			$url  = $this->db->escape_string($avatar->url);
			$type = $this->db->escape_string($avatar->type);
			$dims = $this->db->escape_string($avatar->dimensions);
		}
		else
			$url = $type = $dims = '';
		
		$this->db->write_query('UPDATE `'.TABLE_PREFIX.'users`
			SET `avatar` = "'.$url.'", `avatartype` = "'.$type.'", `avatardimensions` = "'.$dims.'"
			WHERE `uid` = "'.$this->db->escape_string($uid).'"');
	}
	
	/**
	 * Updates given avatar URL for the given user in the MyBB users table
	 *
	 * @param int $uid User ID of the user to update
	 * @param string $avatarUrl Avatar URL to update
	 */
	public function updateProfileAvatarUrl($uid, $avatarUrl)
	{
		$this->db->write_query('UPDATE `'.TABLE_PREFIX.'users`
			SET `avatar` = "'.$this->db->escape_string($avatarUrl).'"
			WHERE `uid` = "'.$this->db->escape_string($uid).'"');
	}
}