<?php
/**
 * This file is part of D3mocracy.
 * 
 * D3mocracy is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * at your option) any later version.

 * D3mocracy is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with D3mocracy.  If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Saša T. <sasa@mindframes.org>
 */

/**
 * Votes DB class
 */
class D3mocracyVotesDB
{
	/**
	 * Database object
	 *
	 * @var mixed
	 */
	private $db;
	
	/**
	 * Constructor; Sets datebase object
	 */
	public function __construct($db)
	{
		$this->db = $db;
	}
	
	/**
	 * Inserts vote, updates existing one or deletes existing one
	 *
	 * @param D3mocracyVote $vote Vote Model of the vote to add, update or remove
	 * @param string $mode Whether to add, update or remove vote (added/updated/removed, respectively)
	 */
	public function vote(D3mocracyVote $vote, $mode)
	{
		if($mode == 'added')
			$sql = 'INSERT INTO `'.TABLE_PREFIX.'d3mocracy_votes`
				(`avatarid`, `voterid`, `type`)
				VALUES
				(
					"'.$this->db->escape_string($vote->avatarid).'",
					"'.$this->db->escape_string($vote->voterid).'",
					"'.$this->db->escape_string($vote->type).'"
				)';
		
		else if($mode == 'updated')
			$sql = 'UPDATE `'.TABLE_PREFIX.'d3mocracy_votes`
				SET `type` = "'.$this->db->escape_string($vote->type).'"
				WHERE `avatarid` = "'.$this->db->escape_string($vote->avatarid).'"
				AND `voterid` = "'.$this->db->escape_string($vote->voterid).'"';
		
		else if($mode == 'removed')
			$sql = 'DELETE FROM `'.TABLE_PREFIX.'d3mocracy_votes`
				WHERE `avatarid` = "'.$this->db->escape_string($vote->avatarid).'"
				AND `voterid` = "'.$this->db->escape_string($vote->voterid).'"';
		
		$this->db->write_query($sql);
	}
	
	/**
	 * Deletes votes of the supplied avatars
	 *
	 * @param array $avatarsToDelete Collection of D3mocracyAvatar instances whose votes are to be deleted
	 */
	public function deleteVotes(array $avatarsToDelete)
	{
		$avatarIDsToDelete = array();
		foreach($avatarsToDelete as $avatar)
			$avatarIDsToDelete[] = $this->db->escape_string($avatar->avatarid);
		
		if(count($avatarIDsToDelete) > 0)
			$this->db->write_query('DELETE
				FROM `'.TABLE_PREFIX.'d3mocracy_votes`
				WHERE `avatarid` IN ("'.join('","', $avatarIDsToDelete).'")');
	}
	
	/**
	 * Recounts votes for avatars starting at given position through given quantity
	 *
	 * @param int $position Start position for the LIMIT clause
	 * @param int $perPage Quantity for the LIMIT clause
	 */
	public function recountVotes($position, $perPage)
	{
		$query = $this->db->query('SELECT `avatarid`
			FROM `'.TABLE_PREFIX.'d3mocracy_avatars`
			ORDER BY `avatarid`
			LIMIT '.$position.','.$perPage);
		
		if($this->db->num_rows($query) > 0)
		{
			$avatarids = '';
			while($avatarid = $this->db->fetch_field($query, 'avatarid'))
				$avatarids .= $avatarid.',';
			$avatarids = substr($avatarids, 0, -1);
			
			$this->db->write_query('
				UPDATE `'.TABLE_PREFIX.'d3mocracy_avatars` `a`
				LEFT JOIN (SELECT `avatarid`, COUNT(*) AS `likes`
					FROM `'.TABLE_PREFIX.'d3mocracy_votes`
					WHERE `type` = "like" AND `avatarid` IN ('.$avatarids.')
					GROUP BY `avatarid`) `l`
				ON `a`.`avatarid` = `l`.`avatarid`
				LEFT JOIN (SELECT `avatarid`, COUNT(*) AS `dislikes`
					FROM `'.TABLE_PREFIX.'d3mocracy_votes`
					WHERE `type` = "dislike" AND `avatarid` IN ('.$avatarids.')
					GROUP BY `avatarid`) `d`
				ON `a`.`avatarid` = `d`.`avatarid`
				SET `a`.`likes` = IFNULL(`l`.`likes`, 0),
					`a`.`dislikes` = IFNULL(`d`.`dislikes`, 0)
				WHERE `a`.`avatarid` IN ('.$avatarids.')');
		}
	}
}