<?php
/**
 * This file is part of D3mocracy.
 * 
 * D3mocracy is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * at your option) any later version.

 * D3mocracy is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with D3mocracy.  If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Saša T. <sasa@mindframes.org>
 */

/**
 * Window Model
 */
class D3mocracyWindow
{
	/**
	 * Collection of D3mocracyAvatar instances
	 *
	 * @var array
	 */
	private $avatars = array();
	
	/**
	 * D3mocracyAvatar instance of the expired active avatar
	 *
	 * @var array
	 */
	private $expiredActiveAvatar;
	
	/**
	 * Collection of D3mocracyAvatar instances containing expired inactive avatars to delete
	 *
	 * @var array
	 */
	private $avatarsToDelete = array();
	
	/**
	 * Classifies avatars as avatars to delete, avatars to deactivate and display-ready avatars
	 *
	 * @param array $avatars Avatars to classify
	 */
	public function pickAvatars(array $avatars)
	{
		foreach($avatars as $avatar)
		{
			// Active and expired
			if($avatar->isActiveAndExpired())
			{
				$this->expiredActiveAvatar = $avatar;
				$this->addAvatar($this->expiredActiveAvatar);
			}
			
			// Inactive and expired - add to delete queue
			else if($avatar->isInactiveAndExpired())
				$this->avatarsToDelete[] = $avatar;
			
			// Display-ready
			else
				$this->addAvatar($avatar);
		}
	}
	
	/**
	 * Adds an Avatar Model instance to the collection
	 *
	 * @param D3mocracyAvatar $avatar Avatar Model instance to add
	 */
	public function addAvatar(D3mocracyAvatar $avatar)
	{
		$this->avatars[] = $avatar;
	}
	
	/**
	 * Removes an Avatar Model instance from the collection
	 *
	 * @param D3mocracyAvatar $avatar Avatar Model instance to remove
	 */
	public function removeAvatar(D3mocracyAvatar $avatar)
	{
		$key = array_search($avatar, $this->avatars, true);
		if($key !== false)
			unset($this->avatars[$key]);
	}
	
	/**
	 * Returns Avatar Model instance collection
	 *
	 * @return array Array with D3mocracyAvatar instances
	 */
	public function getAvatars()
	{
		return $this->avatars;
	}
	
	/**
	 * Returns D3mocracyAvatar instance of the expired active avatar
	 *
	 * @return D3mocracyAvatar Expired active avatar
	 */
	public function getExpiredActiveAvatar()
	{
		return $this->expiredActiveAvatar;
	}
	
	/**
	 * Removes D3mocracyAvatar instance of the expired active avatar locally
	 */
	public function removeExpiredActiveAvatar()
	{
		$this->expiredActiveAvatar = null;
	}
	
	/**
	 * Adds a D3mocracyAvatar instance to the delete list
	 *
	 * @param D3mocracyAvatar $avatar Avatar Model instance to add
	 */
	public function addAvatarToDelete(D3mocracyAvatar $avatar)
	{
		$this->avatarsToDelete[] = $avatar;
	}
	
	/**
	 * Returns D3mocracyAvatar instances containing avatars to delete
	 *
	 * @return array Expired inactive avatars
	 */
	public function getAvatarsToDelete()
	{
		return $this->avatarsToDelete;
	}
	
	/**
	 * Removes D3mocracyAvatar instances containing avatars to delete locally
	 */
	public function removeAvatarsToDelete()
	{
		$this->avatarsToDelete = array();
	}
	
	/**
	 * Returns avatar count based on number of Avatar Model instances
	 *
	 * @return int Avatar count
	 */
	public function getNumAvatars()
	{
		return count($this->avatars);
	}
	
	/**
	 * Returns Avatar Model instance by avatarid
	 *
	 * @param int $avatarid avatarid of the avatar to return
	 * @return D3mocracyAvatar Avatar Model, or null if it's not found
	 */
	public function getAvatarByID($avatarid)
	{
		foreach($this->avatars as $avatar)
			if($avatar->avatarid == $avatarid)
				return $avatar;
		
		return null;
	}
	
	/**
 	 * Returns ID of the currently active avatar
 	 *
 	 * @return int Active avatar ID or null if there are no avatars
 	 */
	public function getActiveAvatar()
	{
		if($this->getNumAvatars())
		{
			$this->avatars = array_values($this->avatars);
			return $this->avatars[0];
		}
		else
			return null;
	}
	
	/**
	 * Sorts avatar collection
	 */
	public function sortAvatars()
	{
		usort($this->avatars, array($this, 'sortAvatarsByRatingSubmitted'));
	}
	
	/**
	 * Avatar sorting callback method; Sorts by rating (descending) and submission timestamp (ascending)
	 *
	 * @param D3mocracyAvatar $avatar1 Avatar to compare against the other one
	 * @param D3mocracyAvatar $avatar2 Avatar to compare against the other one
	 * @return int 1 if first avatar is greater than second, -1 if it's lower, 0 if they're equal
	 */
	public function sortAvatarsByRatingSubmitted(D3mocracyAvatar $avatar1, D3mocracyAvatar $avatar2)
	{
		$avatar1Rating = $avatar1->getRating();
		$avatar2Rating = $avatar2->getRating();
		
		if($avatar1Rating < $avatar2Rating)
			return  1;
		else if($avatar1Rating > $avatar2Rating)
			return -1;
		else if($avatar1->submitted < $avatar2->submitted)
			return -1;
		else if($avatar1->submitted > $avatar2->submitted)
			return  1;
		else
			return  0;
	}
}