<?php
/**
 * This file is part of D3mocracy.
 * 
 * D3mocracy is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * at your option) any later version.

 * D3mocracy is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with D3mocracy.  If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Saša T. <sasa@mindframes.org>
 */

/**
 * Window actions class
 */
class D3mocracyWindowActions
{
	/**
	 * Window Model instance
	 *
	 * @var D3mocracyWindow
	 */
	public $window;
	
	/**
	 * Constructor; Sets window Model instance
	 *
	 * @param D3mocracyWindow $window Window Model
	 */
	public function __construct(D3mocracyWindow $window)
	{
		$this->window = $window;
	}
	
	/**
	 * Filters expired avatars and deletes or deactivates them; Returns new active avatar
	 *
	 * @param D3mocracyOwner $owner Owner Model instance
	 * @param D3mocracyAvatarsDB $dbAvatars Avatars DB object
	 * @param D3mocracyVotesDB $dbVotes Votes DB object
	 * @param D3mocracyUsersDB $dbUsers Users DB object
	 * @param bool &$profileAvatarUpdated Whether active avatar was updated
	 * @param bool $dontUpdateProfile If set to true, profile avatar won't be updated
	 * @return D3mocracyAvatar Avatar Model instance of the active avatar, null if there are no avatars
	 */
	public function handleExpiredAvatars
	(
		D3mocracyOwner $owner,
		D3mocracyAvatarsDB $dbAvatars,
		D3mocracyVotesDB $dbVotes,
		D3mocracyUsersDB $dbUsers,
		&$profileAvatarUpdated,
		$dontUpdateProfile = false
	){
		// Profile avatar updated - true/false
		$profileAvatarUpdated = false;
		
		// Deactivate expired active avatar
		if($this->window->getExpiredActiveAvatar())
		{
			$expiredActiveAvatar = $this->window->getExpiredActiveAvatar();
			$expiredActiveAvatar->deactivateActive();
			$expiredActiveAvatarSince = $expiredActiveAvatar->since;
			$this->window->sortAvatars();
			
			// Expired-active is active again
			if($this->window->getActiveAvatar() === $expiredActiveAvatar)
			{
				// No need to set profile avatar because it's already set
				$expiredActiveAvatar->loopActivate();
				$dbVotes->deleteVotes(array($expiredActiveAvatar));
				$dbAvatars->reactivateResetVoteCache($expiredActiveAvatar);
			}
			
			// Expired-active is now expired-inactive
			else if($expiredActiveAvatar->isInactiveAndExpired())
			{
				// Profile avatar will be reset below (*1)
				$this->window->removeAvatar($expiredActiveAvatar);
				$this->window->addAvatarToDelete($expiredActiveAvatar);
			}
			
			// Expired-active is inactive
			else
			{
				// Profile avatar will be reset below (*1)
				$dbVotes->deleteVotes(array($expiredActiveAvatar));
				$dbAvatars->reactivateResetVoteCache($expiredActiveAvatar);
			}
			$this->window->removeExpiredActiveAvatar();
			
			// Expired-active avatar was deactivated, found new active avatar
			$activeAvatar = $this->window->getActiveAvatar();
			if($activeAvatar != null && $activeAvatar->status == 'inactive')
			{
				// Reseting active avatar (^1)
				$activeAvatar->loopActivate($expiredActiveAvatarSince);
				if(!$dontUpdateProfile)
				{
					$dbAvatars->updateActiveAvatar($activeAvatar->avatarid, $activeAvatar->since);
					$owner->setAvatarParams($activeAvatar);
					$profileAvatarUpdated = true;
				}
				else
					$dbAvatars->updateActiveAvatarOnly($activeAvatar->avatarid, $activeAvatar->since);
			}
		}
		
		// Remove expired inactive avatars
		if(count($this->window->getAvatarsToDelete()) > 0)
		{
			$dbAvatars->deleteAvatarsVotes($this->window->getAvatarsToDelete());
			$dbUsers->recalcAvatarCount($owner->uid);
			$this->window->removeAvatarsToDelete();
		}
		
		return $this->window->getActiveAvatar();
	}
}