<?php
/**
 * This file is part of D3mocracy.
 * 
 * D3mocracy is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * at your option) any later version.

 * D3mocracy is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with D3mocracy.  If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Saša T. <sasa@mindframes.org>
 */

/**
 * Window View Component
 */
class D3mocracyWindowView
{
	/**
	 * Default avatar box dimension
	 */
	const DEFAULT_DIMENSION = 120;
	
	/**
	 * Dimension padding to be user in avatar box body
	 */
	const DIMENSION_PADDING = 8;
	
	/**
	 * Distance between avatar boxes
	 */
	const BOX_PADDING = 10;
	
	/**
	 * D3mocracyWindow instance
	 *
	 * @var D3mocracyWindow
	 */
	private $window;
	
	/**
	 * Collection of D3mocracyAvatarView instances
	 *
	 * @var array
	 */
	private $avatarViews = array();
	
	/**
	 * D3mocracyAvatarNewView instance
	 *
	 * @var D3mocracyAvatarNewView
	 */
	private $avatarNewView;
	
	/**
	 * Stored highest dimension property
	 *
	 * @var int
	 */
	private $highestDimension;
	
	/**
	 * Stored avatar box's head and foot width property
	 *
	 * @var int
	 */
	private $headFootWidth;
	
	/**
	 * Constructor; Sets window Model instance
	 *
	 * @param D3mocracyWindow $window Window Model
	 */
	public function __construct(D3mocracyWindow $window)
	{
		$this->window = $window;
	}
	
	/**
	 * Adds avatar View Component Model
	 *
	 * @param D3mocracyAvatarView $avatarView Avatar View Component
	 */
	public function addAvatarView(D3mocracyAvatarView $avatarView)
	{
		$this->avatarViews[] = $avatarView;
	}
	
	/**
	 * Adds new avatar View Component Model
	 *
	 * @param D3mocracyAvatarNewView $avatarView New avatar View Component Model
	 */
	public function addAvatarNewView(D3mocracyAvatarNewView $avatarNewView)
	{
		$this->avatarNewView = $avatarNewView;
	}
	
	/**
	 * Returns Avatar View Components
	 *
	 * @return array Collection of Avatar View Components
	 */
	public function getAvatarViews()
	{
		return $this->avatarViews;
	}
	
	/**
	 * Recalculates and stores highest dimension and avatar box head/foot width
	 */
	public function recalculate()
	{
		$dimension = self::DEFAULT_DIMENSION;
		foreach($this->avatarViews as $view)
		{
			if(($viewWidth = $view->getViewWidth()) > $dimension)
				$dimension = $viewWidth;
			
			if(($viewHeight = $view->getViewHeight()) > $dimension)
				$dimension = $viewHeight;
		}
		$this->highestDimension = $dimension + self::DIMENSION_PADDING;
		$this->headFootWidth = $this->highestDimension - D3MOCRACY_BOX_WIDTH_PADDING;
	}
	
	/**
	 * Returns window wide highest avatar box dimension
	 *
	 * @return int Highest avatar box dimension
	 */
	public function getHighestDimension()
	{
		return $this->highestDimension;
	}
	
	/**
	 * Returns avatar box head's and foot's stored width
	 *
	 * @return int Head/foot width
	 */
	public function getBoxHeadFootWidth()
	{
		return $this->headFootWidth;
	}
	
	/**
	 * Returns avatar box body's stored width/height
	 *
	 * @return int Body width/height
	 */
	public function getBoxBodyWidthHeight()
	{
		return $this->highestDimension;
	}
	
	/**
	 * Returns avatar box count for the window
	 *
	 * @return int Avatar box count
	 */
	public function getNumBoxes()
	{
		$numBoxes  = $this->window->getNumAvatars();
		$numBoxes += $this->avatarNewView->isVisible() ? 1 : 0;
		return $numBoxes;
	}
	
	/**
	 * Returns width CSS property to be used on window div
	 *
	 * @return string Width CSS property
	 */
	public function getWidthStyle()
	{
		$numBoxes = $this->getNumBoxes();
		$dimension = $this->getHighestDimension();
		$padding = self::BOX_PADDING;
		$width = 200;
		
		if($numBoxes <= D3MOCRACY_WINDOW_BOX_WRAP)
		{
			$width = ($numBoxes * $dimension) + ($numBoxes * $padding);
		}
		else if($numBoxes > D3MOCRACY_WINDOW_BOX_WRAP)
		{
			$width = (D3MOCRACY_WINDOW_BOX_WRAP * $dimension) + (D3MOCRACY_WINDOW_BOX_WRAP * $padding);
		}
		
		return "width:{$width}px;";
	}
	
	/**
	 * Renders window's template and returns output
	 *
	 * @param templates $templates Instance of templates, MyBB templating object
	 * @return string Rendered window's HTML
	 */
	public function render(templates $templates, D3mocracyOwner $owner, MyLanguage $lang)
	{
		$avatarBoxes = '';
		$windowView = $this;
		
		// Get output
		foreach($this->avatarViews as $avatarView)
			$avatarBoxes .= $avatarView->render($templates, $this);
		$avatarBoxes .= $this->avatarNewView->render($templates, $this, $lang);
		eval('$window = "'.$templates->get("d3mocracy_window").'";');
		
		// Return output
		return $window;
	}
}