<?php
/**
 * This file is part of D3mocracy.
 * 
 * D3mocracy is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * at your option) any later version.

 * D3mocracy is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with D3mocracy.  If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Saša T. <sasa@mindframes.org>
 */

// Disallow direct access to this file for security reasons
if (!defined ("IN_MYBB"))
{
	die ("Direct initialization of this file is not allowed.<br /><br />Please make sure IN_MYBB is defined.");
}

/**
 * Loads D3mocracy class-files
 */
function d3mocracyLoadClasses()
{
	require_once MYBB_ROOT."inc/plugins/d3mocracy/classes/D3mocracyModel.php";
	require_once MYBB_ROOT."inc/plugins/d3mocracy/classes/D3mocracyAvatarActions.php";
	require_once MYBB_ROOT."inc/plugins/d3mocracy/classes/D3mocracyAvatarNewView.php";
	require_once MYBB_ROOT."inc/plugins/d3mocracy/classes/D3mocracyAvatar.php";
	require_once MYBB_ROOT."inc/plugins/d3mocracy/classes/D3mocracyAvatarsDB.php";
	require_once MYBB_ROOT."inc/plugins/d3mocracy/classes/D3mocracyAvatarView.php";
	require_once MYBB_ROOT."inc/plugins/d3mocracy/classes/D3mocracyAvatarVoteDiff.php";
	require_once MYBB_ROOT."inc/plugins/d3mocracy/classes/D3mocracyOutputPrinter.php";
	require_once MYBB_ROOT."inc/plugins/d3mocracy/classes/D3mocracyOwner.php";
	require_once MYBB_ROOT."inc/plugins/d3mocracy/classes/D3mocracyRequester.php";
	require_once MYBB_ROOT."inc/plugins/d3mocracy/classes/D3mocracySubmitFormView.php";
	require_once MYBB_ROOT."inc/plugins/d3mocracy/classes/D3mocracySubmittedView.php";
	require_once MYBB_ROOT."inc/plugins/d3mocracy/classes/D3mocracyUsersDB.php";
	require_once MYBB_ROOT."inc/plugins/d3mocracy/classes/D3mocracyVote.php";
	require_once MYBB_ROOT."inc/plugins/d3mocracy/classes/D3mocracyVotesDB.php";
	require_once MYBB_ROOT."inc/plugins/d3mocracy/classes/D3mocracyWindowActions.php";
	require_once MYBB_ROOT."inc/plugins/d3mocracy/classes/D3mocracyWindow.php";
	require_once MYBB_ROOT."inc/plugins/d3mocracy/classes/D3mocracyWindowView.php";
}

/**
 * Returns D3mocracy template names
 *
 * @return array An array of template names used by D3mocracy
 */
function d3mocracyGetTemplateNames()
{
	return array
	(
		'd3mocracy_page',
		'd3mocracy_header',
		'd3mocracy_usercp_script',
		'd3mocracy_window',
		'd3mocracy_empty_avatar',
		'd3mocracy_avatar_box_hidden_container',
		'd3mocracy_avatar_add',
		'd3mocracy_avatar_add_script',
		'd3mocracy_avatar_box',
		'd3mocracy_avatar_box_new',
		'd3mocracy_avatar_count',
		'd3mocracy_delete_button'
	);
}

/**
 * Inserts MyBB template modifications and D3mocracy templates
 */
function d3mocracyInstallTemplates()
{
	global $db;
	
	foreach(array('postbit', 'postbit_classic') as $tplTitle)
	{
		find_replace_templatesets
		(
			$tplTitle,
			'#'.preg_quote('{$post[\'useravatar\']}').'#i',
			'<span class="d3mocracy_active_avatar">'.
				'<input type="hidden" class="d3mocracy_status" value="{$post[\'d3mocracystatus\']}" />'.
				'<input type="hidden" class="d3mocracy_uid" value="{$post[\'uid\']}" />'.
				'{$post[\'d3mocracyemptyavatar\']}{$post[\'useravatar\']}</span>'
		);
	}
	find_replace_templatesets
	(
		'headerinclude',
		'#'.preg_quote('{$stylesheets}').'#i',
		'{$stylesheets}'."\n".'{$mybb->d3mocracyHeader}'
	);
	find_replace_templatesets
	(
		'postbit_author_user',
		'#'.preg_quote('{$post[\'postnum\']}<br />').'#i',
		'\\0'."\n\t".'{$post[\'avatarcount\']}'
	);
	find_replace_templatesets
	(
		'postbit_avatar',
		'#'.preg_quote('{$avatar_width_height}').'#i',
		'{$avatar_width_height} class="showthread_avatar_img"'
	);
	find_replace_templatesets
	(
		'usercp_options',
		'@(\{\$lang\->other_options\}\s*[\w\W]+<table[\w\s\-\[\]\'"=]*>\s*<tr>)@U',
		'$1
<td valign="top" width="1"><input type="checkbox" class="checkbox" name="d3mocracydisablesubmissions" id="d3mocracydisablesubmissions" value="1" {$mybb->d3mocracyOptions[\'disablecheck\']} /></td>
<td><span class="smalltext"><label for="d3mocracydisablesubmissions">{$lang->d3mocracy_disable_submissions}</label></span></td>
</tr>
<tr>');
	
	foreach(d3mocracyGetTemplateNames() as $tplName)
	{
		$templateBody = file_get_contents(MYBB_ROOT."inc/plugins/d3mocracy/install/templates/".$tplName);
		$db->insert_query
		(
			'templates', array
			(
				'title' => $tplName,
				'template' => $db->escape_string($templateBody),
				'version' => D3MOCRACY_INT_VERSION,
				'dateline' => TIME_NOW,
				'sid' => '-1',
			)
		);
	}
}

/**
 * Cleans MyBB template modifications and deletes D3mocracy templates
 */
function d3mocracyUninstallTemplates()
{
	global $db;
	
	foreach(array('postbit', 'postbit_classic') as $tplTitle)
	{
		find_replace_templatesets
		(
			$tplTitle,
			'#'.preg_quote('<span class="d3mocracy_active_avatar">'.
				'<input type="hidden" class="d3mocracy_status" value="{$post[\'d3mocracystatus\']}" />'.
				'<input type="hidden" class="d3mocracy_uid" value="{$post[\'uid\']}" />'.
				'{$post[\'d3mocracyemptyavatar\']}{$post[\'useravatar\']}</span>').'#i',
			'{$post[\'useravatar\']}'
		);
	}
	find_replace_templatesets
	(
		'headerinclude',
		'#'.preg_quote('{$stylesheets}').'\s*'.preg_quote('{$mybb->d3mocracyHeader}').'#i',
		'{$stylesheets}'
	);
	find_replace_templatesets
	(
		'postbit_author_user',
		"#\s*".preg_quote('{$post[\'avatarcount\']}')."#i",
		''
	);
	find_replace_templatesets
	(
		'postbit_avatar',
		'#'.preg_quote('{$avatar_width_height}').'\s*'.preg_quote('class="showthread_avatar_img"').'#i',
		'{$avatar_width_height}'
	);
	find_replace_templatesets
	(
		'usercp_options',
		'#'.preg_quote('<tr>').'\s*'.
			preg_quote('<td valign="top" width="1"><input type="checkbox" class="checkbox" name="d3mocracydisablesubmissions" id="d3mocracydisablesubmissions" value="1" {$mybb->d3mocracyOptions[\'disablecheck\']} /></td>').'\s*'.
			preg_quote('<td><span class="smalltext"><label for="d3mocracydisablesubmissions">{$lang->d3mocracy_disable_submissions}</label></span></td>').'\s*'.
			preg_quote('</tr>').'\s*#',
		''
	);
	
	foreach(d3mocracyGetTemplateNames() as $tplName)
	{
		$db->write_query("DELETE FROM `".TABLE_PREFIX."templates` WHERE `title` IN ('{$tplName}')");
	}
}

/**
 * Deletes D3mocracy uploaded avatars
 */
function d3mocracyDeleteFiles()
{
	global $mybb;
	$dir = MYBB_ROOT.'/'.d3mocracyGetUploadDirName();
	$dirHandle = opendir($dir);
	while($file = readdir($dirHandle))
	{
		if($file != '.' && $file != '..')
		{
			@unlink("$dir/$file");
		}
	}
}

/**
 * Checks whether D3mocracy avatar dir is writable
 *
 * @return bool True if it's writable, false otherwise
 */
function d3mocracyDirIsWritable()
{
	global $mybb;
	$dir = MYBB_ROOT.'/'.d3mocracyGetUploadDirName();
	return file_exists($dir) && is_dir($dir) && is_writable($dir);
}

/**
 * Creates D3mocracy tables
 */
function d3mocracyInstallDB()
{
	global $db;
	
	$db->write_query("CREATE TABLE `".TABLE_PREFIX."d3mocracy_avatars` (
		`avatarid` int UNSIGNED NOT NULL auto_increment,
		`ownerid` int UNSIGNED NOT NULL DEFAULT '0',
		`submitterid` int UNSIGNED NOT NULL DEFAULT '0',
		`type` varchar(10) NOT NULL DEFAULT '',
		`dimensions` varchar(10) NOT NULL DEFAULT '',
		`url` varchar(200) NOT NULL DEFAULT '',
		`likes` int SIGNED NOT NULL DEFAULT '0',
		`dislikes` int SIGNED NOT NULL DEFAULT '0',
		`status` enum('inactive', 'active') NOT NULL DEFAULT 'inactive',
		`submitted` bigint(30) UNSIGNED NOT NULL DEFAULT '0',
		`since` bigint(30) UNSIGNED NOT NULL DEFAULT '0',
		PRIMARY KEY (`avatarID`),
		KEY(`ownerid`),
		KEY(`submitterid`),
		KEY(`status`),
		KEY(`submitted`),
		KEY(`since`)
	) ENGINE=MyISAM ".$db->build_create_table_collation());
	
	$db->write_query("CREATE TABLE `".TABLE_PREFIX."d3mocracy_votes` (
		`avatarid` int UNSIGNED NOT NULL DEFAULT '0',
		`voterid` int UNSIGNED NOT NULL DEFAULT '0',
		`type` enum('like', 'dislike') DEFAULT 'like',
		KEY(`avatarid`),
		KEY(`voterid`),
		KEY(`type`)
	) ENGINE=MyISAM ".$db->build_create_table_collation());
	
	$db->write_query("CREATE TABLE `".TABLE_PREFIX."d3mocracy_users` (
		`userid` int UNSIGNED NOT NULL DEFAULT '0',
		`status` enum('on','off') NOT NULL DEFAULT 'on',
		`avatarcount` tinyint UNSIGNED NOT NULL DEFAULT '0',
		PRIMARY KEY(`userid`)
	) ENGINE=MyISAM ".$db->build_create_table_collation());
}

/**
 * Drops D3mocracy tables and blanks profile avatar entries of the avatars uploaded through D3mocracy
 */
function d3mocracyUninstallDB()
{
	global $db, $mybb;
	$dirName = d3mocracyGetUploadDirName();
	
	$db->write_query("DROP TABLE IF EXISTS `".TABLE_PREFIX."d3mocracy_avatars`");
	$db->write_query("DROP TABLE IF EXISTS `".TABLE_PREFIX."d3mocracy_votes`");
	$db->write_query("DROP TABLE IF EXISTS `".TABLE_PREFIX."d3mocracy_users`");
	
	// Delete the ones that failed to rename
	$db->write_query('UPDATE `'.TABLE_PREFIX.'users`
		SET `avatar` = "", `avatardimensions` = "", `avatartype` = ""
		WHERE `avatartype` = "upload"
		AND `avatar` LIKE "'.$db->escape_string($dirName).'/%"');
}

/**
 * Inserts d3mocracy setting group and settings
 */
function d3mocracyInsertSettings()
{
	global $db;
	$enLang = d3mocracyGetEnglishLanguageVariable('admin/d3mocracyadmin.lang.php');
	$displayOrder = $db->query('SELECT `disporder` FROM `'.TABLE_PREFIX.'settinggroups` ORDER BY `disporder` DESC LIMIT 1');
	$displayOrder = $db->fetch_array($displayOrder);
	$displayOrder = (int)$displayOrder['disporder'];
	$settingGroup = array
	(
		'name'        => 'd3mocracy',
		'title'       => $enLang['setting_group_d3mocracy'],
		'description' => $enLang['setting_group_d3mocracy_desc'],
		'disporder'   => $displayOrder,
	);
	$db->insert_query('settinggroups', $settingGroup);
	$settingGroupID = $db->insert_id();
	$settings = array
	(
		array
		(
			'name' => 'd3boxwrap',
			'title' => $enLang['setting_d3boxwrap'],
			'description' => $enLang['setting_d3boxwrap_desc'],
			'optionscode' => 'text',
			'value' => '5',
		),
		array
		(
			'name' => 'd3maxavatars',
			'title' => $enLang['setting_d3maxavatars'],
			'description' => $enLang['setting_d3maxavatars_desc'],
			'optionscode' => 'text',
			'value' => '10',
		),
		array
		(
			'name' => 'd3activityts',
			'title' => $enLang['setting_d3activityts'],
			'description' => $enLang['setting_d3activityts_desc'],
			'optionscode' => 'text',
			'value' => '30',
		),
		array
		(
			'name' => 'd3inactivityts',
			'title' => $enLang['setting_d3inactivityts'],
			'description' => $enLang['setting_d3inactivityts_desc'],
			'optionscode' => 'text',
			'value' => '15',
		),
	);
	
	$displayOrder = 0;
	foreach($settings as $setting)
	{
		$setting['gid'] = $settingGroupID;
		$setting['disporder'] = $displayOrder ++;
		$db->insert_query('settings', $setting);
	}
}

/**
 * Removes d3mocracy setting group and settings
 */
function d3mocracyRemoveSettings()
{
	global $db;
	$db->write_query('DELETE `g`, `s`
		FROM `'.TABLE_PREFIX.'settinggroups` `g`
		LEFT JOIN `'.TABLE_PREFIX.'settings` `s`
		ON `g`.`gid` = `s`.`gid`
		WHERE `g`.`name` = "d3mocracy"');
}

/**
 * Returns language variable from the specified english language file
 *
 * @param string $langFile Relative path of the english language file
 * @return array Language variable
 */
function d3mocracyGetEnglishLanguageVariable($langFile)
{
	require MYBB_ROOT.'/inc/languages/english/'.$langFile;
	return $l;
}

/**
 * Returns filename discarding query and path (optionally)
 *
 * @param string $avatarPath Avatar path pointing to filename to return
 * @param bool $baseName Whether to discard path
 * @return string Clean filename
 */
function d3mocracyGetAvatarFileName($avatarPath, $baseName = true)
{
	if($baseName)
		$avatarFile = basename($avatarPath);
	else
		$avatarFile = $avatarPath;
	
	$qMarkPos = strrpos($avatarFile, '?');
	if(is_int($qMarkPos))
		$avatarFile = substr_replace($avatarFile, '', $qMarkPos);
	return $avatarFile;
}

/**
 * Replaces everything in the filename up to the final dot with the defined string
 *
 * @param string $filename Filename to replace
 * @param string $str String to use as replacement
 * @param string New filename with old extension
 */
function d3mocracyFilenameReplaceStr($filename, $str)
{
	return substr_replace($filename, $str, 0, strrpos($filename, '.'));
}

/**
 * Returns D3mocracy upload directory path based on MyBB avatar upload path
 *
 * @return string D3mocracy upload directory path
 */
function d3mocracyGetUploadDirName()
{
	global $mybb;
	$lastChar = $mybb->settings['avataruploadpath'][strlen($mybb->settings['avataruploadpath'])];
	return $mybb->settings['avataruploadpath'] . ($lastChar == '/' ? '' : '/') . 'd3mocracy';
}

/**
 * Returns timestamp from avatar URL contained in the query variable dateline
 *
 * @param string $url URL to extract the timestamp from
 * @return int Avatar's timestamp
 */
function d3mocracyGetTimeFromAvatarUrl($url)
{
	if($dateline = strrchr($url, 'dateline='))
		return (int)substr($dateline, 9);
}

/**
 * Submits avatar and returns submitted avatar info/error; Copied from MyBB
 *
 * @return array Array with two values: Array with submitted avatar's parameters and error message (if submission failed)
 */
function mybbDoAvatar()
{
	global $mybb, $templates, $lang, $title, $charset, $stylesheets;
	$updated_avatar = array();
	
	if($_FILES['avatarupload']['name']) // upload avatar
	{
		// If the user doesn't have permission to upload avatars, exit with error
		if($mybb->usergroup['canuploadavatars'] == 0)
		{
			throw new Exception($lang->d3mocracy_no_perms_upload);
		}
		$avatar = upload_avatar();
		if($avatar['error'])
		{
			$avatar_error = $avatar['error'];
		}
		else
		{
			if($avatar['width'] > 0 && $avatar['height'] > 0)
			{
				$avatar_dimensions = $avatar['width']."|".$avatar['height'];
			}
			$updated_avatar = array(
				"avatar" => $avatar['avatar'].'?dateline='.TIME_NOW,
				"avatardimensions" => $avatar_dimensions,
				"avatartype" => "upload"
			);
		}
	}
	else // remote avatar
	{
		$mybb->input['avatarurl'] = preg_replace("#script:#i", "", $mybb->input['avatarurl']);
		$ext = get_extension($mybb->input['avatarurl']);
	
		// Copy the avatar to the local server (work around remote URL access disabled for getimagesize)
		$file = fetch_remote_file($mybb->input['avatarurl']);
		if(!$file)
		{
			$avatar_error = $lang->error_invalidavatarurl;
		}
		else
		{
			$tmp_name = $mybb->settings['avataruploadpath']."/remote_".md5(random_str());
			$fp = @fopen($tmp_name, "wb");
			if(!$fp)
			{
				$avatar_error = $lang->error_invalidavatarurl;
			}
			else
			{
				fwrite($fp, $file);
				fclose($fp);
				list($width, $height, $type) = @getimagesize($tmp_name);
				@unlink($tmp_name);
				if(!$type)
				{
					$avatar_error = $lang->error_invalidavatarurl;
				}
			}
		}
	
		if(empty($avatar_error))
		{
			if($width && $height && $mybb->settings['maxavatardims'] != "")
			{
				list($maxwidth, $maxheight) = explode("x", my_strtolower($mybb->settings['maxavatardims']));
				if(($maxwidth && $width > $maxwidth) || ($maxheight && $height > $maxheight))
				{
					$lang->error_avatartoobig = $lang->sprintf($lang->error_avatartoobig, $maxwidth, $maxheight);
					$avatar_error = $lang->error_avatartoobig;
				}
			}
		}
	
		if(empty($avatar_error))
		{
			if($width > 0 && $height > 0)
			{
				$avatar_dimensions = intval($width)."|".intval($height);
			}
			$updated_avatar = array(
				"avatar" => $mybb->input['avatarurl'].'?dateline='.TIME_NOW,
				"avatardimensions" => $avatar_dimensions,
				"avatartype" => "remote"
			);
		}
	}
	
	return array($updated_avatar, $avatar_error);
}

/**
 * Normalizes some language entries and returns auto resize message and checkbox for the submit form; Copied from MyBB
 *
 * @return string Auto resize message and checkbox for the submit form
 */
function mybbUploadFormMessages()
{
	global $mybb, $lang;
	
	if($mybb->settings['maxavatardims'] != "")
	{
		list($maxwidth, $maxheight) = explode("x", my_strtolower($mybb->settings['maxavatardims']));
		$lang->avatar_note .= "<br />".$lang->sprintf($lang->avatar_note_dimensions, $maxwidth, $maxheight);
	}
	if($mybb->settings['avatarsize'])
	{
		$maxsize = get_friendly_size($mybb->settings['avatarsize']*1024);
		$lang->avatar_note .= "<br />".$lang->sprintf($lang->avatar_note_size, $maxsize);
	}
	if($mybb->settings['avatarresizing'] == "auto")
	{
		$auto_resize = '<br /><span class="smalltext">'.$lang->d3mocracy_avatar_resized.'</span>'."\n";
	}
	else if($mybb->settings['avatarresizing'] == "user")
	{
		$auto_resize = '<br /><span class="smalltext">'.
			'<input type="checkbox" name="auto_resize" value="1" checked="checked" id="auto_resize" /> '.
			'<label for="auto_resize">'.$lang->d3mocracy_try_resize_avatar.'</label></span>'."\n";
	}
	
	return $auto_resize;
}
