/**
 * This file is part of D3mocracy.
 * 
 * D3mocracy is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * at your option) any later version.

 * D3mocracy is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with D3mocracy.  If not, see <http://www.gnu.org/licenses/>.
 *
 * @author Saša T. <sasa@mindframes.org>
 */

var DAvatarBoxPrototype = function(dAvatarBoxDiv, stAvatarLink, dWindow)
{
	this.construct = function(dAvatarBoxDiv, stAvatarLink, dWindow)
	{
		this.dAvatarBoxDiv = dAvatarBoxDiv;
		this.stAvatarLink = stAvatarLink;
		this.dWindow = dWindow;
		
		this.initParams();
		this.initDeleteButton();
		this.initVoteButtons();
		this.middleAlignAvatar();
		this.initTooltip();
		this.setSubmitterSpanWidths();
	}
	
	this.initParams = function()
	{
		this.avatarImg = this.dAvatarBoxDiv.select('.d3mocracy_avatar_body img')[0];
		this.stAvatarImg = this.stAvatarLink.select('img')[0];
		
		this.avatarID = this.dAvatarBoxDiv.select('input.d3mocracy_avatarid')[0].value;
		this.avatarSubmitted = this.dAvatarBoxDiv.select('input.d3mocracy_submitted')[0].value;
		
		this.divHead = this.dAvatarBoxDiv.select('.d3mocracy_avatar_head')[0];
		this.divBody = this.dAvatarBoxDiv.select('.d3mocracy_avatar_body')[0];
		this.divFoot = this.dAvatarBoxDiv.select('.d3mocracy_avatar_foot')[0];
		
		this.divTip = this.divBody.select('.d3mocracy_tip')[0];
		this.divTip.remove();
		this.divTipSubmitted = this.divTip.select('.d3mocracy_tip_submitted')[0];
		this.divTipSince = this.divTip.select('.d3mocracy_tip_since')[0];
		this.divTipExpires = this.divTip.select('.d3mocracy_tip_expires')[0];
		
		var votes = this.dAvatarBoxDiv.select('input.d3mocracy_votes')[0].value.split(',');
		this.avatarLikes = parseInt(votes[0]);
		this.avatarDislikes = parseInt(votes[1]);
		this.avatarRating = this.avatarLikes - this.avatarDislikes;
	}
	
	this.initDeleteButton = function()
	{
		var _this = this;
		var avatarDeleteButton = this.dAvatarBoxDiv.select('a.d3mocracy_delete_avatar');
		if(!avatarDeleteButton.length)
		{
			return;
		}
		this.avatarDeleteButton = avatarDeleteButton[0];
		this.avatarDeleteButton.observe('click', function()
		{
			// Confirm avatar deletion, exit if cancel is clicked
			if(!confirm(d3mocracyLangDeleteConfirm))
			{
				return;
			}
			
			// Suspend event and init loading background image
			this.suspendedEvents = DEventStatic.suspendGet(this, 'click');
			this.addClassName('d3mocracy_deleting_avatar');
			
			// Request avatar deletion
			new Ajax.Request('d3mocracy.php?action=deleteavatar&req='+Math.random(),
			{
				method: 'post',
				parameters: {avatarid: this.rel, csrftoken: my_post_key},
				onSuccess: function(response) {_this.deleteRequestSuccess(response);},
				onFailure: function(response) {_this.deleteRequestFailure();}
			});
		});
	}
	
	this.deleteRequestSuccess = function(response)
	{
		// If error, alert and remove button
		if(response.responseJSON.error)
		{
			this.avatarDeleteButton.remove();
			alert(response.responseJSON.error.message);
		}
		// If success, remove image
		else if(response.responseJSON.deleted)
		{
			// Show "Add New" if requester is deleting the image that he submitted
			if(this.dAvatarBoxDiv.hasClassName('d3mocracy_submitted_by_me'))
			{
				this.dWindow.showAvatarBoxNew();
			}
			
			// Remove and recalculate
			this.dAvatarBoxDiv.remove();
			this.dWindow.removeAvatarBox(this);
			this.dWindow.recalcWindowWidth();
			this.dWindow.updateTooltips(response.responseJSON.avatars);
			DShowthreadStatic.updateAvatarCount(this.dWindow);
			DShowthreadStatic.resetActiveAvatar(this.dWindow);
		}
	}
	
	this.deleteRequestFailure = function()
	{
		// If failed, restore button
		DEventStatic.resume(this.avatarDeleteButton, 'click', this.avatarDeleteButton.suspendedEvents);
		this.avatarDeleteButton.removeClassName('d3mocracy_deleting_avatar');
		alert(d3mocracyLangRequestFailed);
	}
	
	this.initVoteButtons = function()
	{
		var _this = this;
		
		var buttons = this.dAvatarBoxDiv.select
		(
			'.d3mocracy_vote_like',
			'.d3mocracy_vote_dislike',
			'.d3mocracy_voted_like',
			'.d3mocracy_voted_dislike'
		);
		this.avatarLk = buttons[0];
		this.avatarLk.type = 'like';
		this.avatarDk = buttons[1];
		this.avatarDk.type = 'dislike';
		this.ratingSpan = this.dAvatarBoxDiv.select
		(
			'.d3mocracy_rating_positive',
			'.d3mocracy_rating_neutral',
			'.d3mocracy_rating_negative'
		)[0];
		
		[this.avatarLk, this.avatarDk].each(function(avatarButton)
		{
			if(d3mocracyUserRole == 'none')
			{
				// Disable button
				avatarButton.style.backgroundPosition = '0 0';
				avatarButton.style.cursor = 'default';
				avatarButton.title = '';
			}
			else if(d3mocracyUserRole == 'exd3mocrat')
			{
				avatarButton.observe('click', function()
				{
					alert(d3mocracyActionsSuspended);
				});
			}
			else
			{
				avatarButton.observe('click', function()
				{
					_this.suspendVoteEvents();
					this.addClassName('d3mocracy_voting');
					
					// Request avatar deletion
					new Ajax.Request('d3mocracy.php?action=voteavatar&req='+Math.random(),
					{
						method: 'post',
						parameters: {avatarid: _this.avatarID, avatarvote: this.type, csrftoken: my_post_key},
						onSuccess: function(response) {_this.voteRequestSuccess(avatarButton, response);},
						onFailure: function(response) {_this.voteRequestFailure(avatarButton);}
					});
				});
			}
		});
	}
	
	this.voteRequestSuccess = function(avatarButton, response)
	{
		this.resumeVoteEvents();
		avatarButton.removeClassName('d3mocracy_voting');
		if(response.responseJSON)
		{
			var json = response.responseJSON;
		}
		else return;
		
		if(json.error)
		{
			alert(json.error.message);
		}
		else if(json.action)
		{
			this.addRemoveAvatarVote(json);
			this.dWindow.repositionAvatarBox(this);
			this.dWindow.updateTooltips(json.avatars);
			DShowthreadStatic.resetActiveAvatar(this.dWindow);
		}
	}
	
	this.voteRequestFailure = function(avatarButton)
	{
		this.resumeVoteEvents();
		avatarButton.removeClassName('d3mocracy_voting');
		alert(d3mocracyLangRequestFailed);
	}
	
	this.middleAlignAvatar = function()
	{
		var _this = this;
		
		this.dAvatarBoxDiv.select('.d3mocracy_avatar_body img').each(function(item)
		{
			item.style.marginTop = (Math.floor((_this.dWindow.bodyDimension - parseInt(item.style.height)) / 2)) + 'px';
		});
	}
	
	this.initTooltip = function()
	{
		var _this = this;
		var tipTimeout = null;
		
		this.avatarImg.observe('mousemove', function(mouseEvent)
		{
			clearTimeout(tipTimeout);
			var mouseEventClone = Object.clone(mouseEvent); // Fucking IE...
			var timeoutFunction = function() {_this.tooltipShow(mouseEventClone);}
			tipTimeout = setTimeout(timeoutFunction, 1000);
		});
		this.avatarImg.observe('mouseout', function(mouseEvent)
		{
			clearTimeout(tipTimeout);
			_this.tooltipHide(mouseEvent);
		});
	}
	
	this.tooltipShow = function(mouseEvent)
	{
		var element = mouseEvent.element();
		element.tipVisible = true;
		
		this.divTip.style.top  = (mouseEvent.pointerY() + 10) + 'px';
		this.divTip.style.left = (mouseEvent.pointerX() + 10) + 'px';
		$(document.body).insert({'top': this.divTip});
		this.divTip.show();
	}
	
	this.tooltipHide = function(mouseEvent)
	{
		var element = mouseEvent.element();
		if(element.tipVisible)
		{
			this.divTip.hide();
			this.divTip.remove();
		}
		element.tipVisible = false;
	}
	
	this.setSubmitterSpanWidths = function()
	{
		var _this = this;
		
		this.dAvatarBoxDiv.select('.d3mocracy_submitter').each(function(item)
		{
			var dLinks = _this.dAvatarBoxDiv.select('a.d3mocracy_delete_avatar');
			item.style.width = (dLinks.length > 0 ? _this.dWindow.headFootWidth - 14 : _this.dWindow.headFootWidth) + 'px';
		});
	}
	
	this.resetDimensions = function()
	{
		this.divHead.style.width = this.dWindow.headFootWidth + 'px';
		this.divFoot.style.width = this.dWindow.headFootWidth + 'px';
		
		this.divBody.style.width = this.dWindow.bodyDimension + 'px';
		this.divBody.style.height = this.dWindow.bodyDimension + 'px';
		
		this.middleAlignAvatar();
		this.setSubmitterSpanWidths();
	}
	
	this.suspendVoteEvents = function()
	{
		this.avatarLk.suspendedEvents = DEventStatic.suspendGet(this.avatarLk, 'click');
		this.avatarDk.suspendedEvents = DEventStatic.suspendGet(this.avatarDk, 'click');
	}
	
	this.resumeVoteEvents = function()
	{
		DEventStatic.resume(this.avatarLk, 'click', this.avatarLk.suspendedEvents);
		DEventStatic.resume(this.avatarDk, 'click', this.avatarDk.suspendedEvents);
	}
	
	this.addRemoveAvatarVote = function(json)
	{
		// Set counts
		this.avatarLikes = json.likes;
		this.avatarDislikes = json.dislikes;
		this.avatarLk.select('.d3mocracy_num_votes')[0].innerHTML = this.avatarLikes + '';
		this.avatarDk.select('.d3mocracy_num_votes')[0].innerHTML = this.avatarDislikes + '';
		this.avatarRating = this.avatarLikes - this.avatarDislikes;
		this.ratingSpan.innerHTML = this.avatarRating + '';
		
		// Set class names and titles
		this.avatarLk.className = json.classLk;
		this.avatarDk.className = json.classDk;
		this.avatarLk.title = json.titleLk;
		this.avatarDk.title = json.titleDk;
		this.ratingSpan.className = json.classRating;
	}
	
	this.construct(dAvatarBoxDiv, stAvatarLink, dWindow);
}

var DWindowPrototype = function(dWindowDiv, stAvatarLink, uid)
{
	this.construct = function(dWindowDiv, stAvatarLink, uid)
	{
		this.dWindowDiv = dWindowDiv;
		this.stAvatarLink = stAvatarLink;
		this.uid = uid;
		this.dAvatarBoxes = {};
		this.initParams();
		this.initHideButton();
		this.initAvatarBoxes();
		DShowthreadStatic.updateAvatarCount(this);
		DShowthreadStatic.resetActiveAvatar(this);
		this.dWindowDiv.show();
	}
	
	this.getDAvatarBox = function(dAvatarBoxDiv)
	{
		for(var a in this.dAvatarBoxes)
		{
			if(this.dAvatarBoxes[a].dAvatarBoxDiv === dAvatarBoxDiv)
			{
				return this.dAvatarBoxes[a];
			}
		}
		return false;
	}
	
	this.createAvatarBox = function(dAvatarBoxDiv)
	{
		var dAvatarBox = new DAvatarBoxPrototype(dAvatarBoxDiv, stAvatarLink, this);
		this.dAvatarBoxes[dAvatarBox.avatarID] = dAvatarBox;
		return dAvatarBox;
	}
	
	this.insertNewAvatarBox = function(dAvatarBoxHtml)
	{
		this.hideAvatarBoxNew();
		this.avatarBoxNewDiv.insert({'before': dAvatarBoxHtml});
		var insertedAvatarBoxDiv = this.avatarBoxNewDiv.previous('.d3mocracy_avatar_box');
		var insertedAvatarDimension = parseInt(insertedAvatarBoxDiv.select('.d3mocracy_avatar_body')[0].style.height);
		var insertedAvatarBox = this.createAvatarBox(insertedAvatarBoxDiv);
		this.numBoxes ++;
		
		// Applying new dimensions to all window avatars
		if(insertedAvatarDimension > this.bodyDimension)
		{
			// Reset avatar dimension properties in this window and in Add New link
			this.resetDimensions(insertedAvatarDimension);
			this.resetAddNewBoxDimensions();
			
			for(var aID in this.dAvatarBoxes)
			{
				this.dAvatarBoxes[aID].resetDimensions();
			}
		}
		// Applying window-wide dimensions to new avatar
		else if(insertedAvatarDimension < this.bodyDimension)
		{
			insertedAvatarBox.resetDimensions();
		}
		
		this.repositionAvatarBox(insertedAvatarBox);
		DShowthreadStatic.updateAvatarCount(this);
		DShowthreadStatic.resetActiveAvatar(this);
		this.recalcWindowWidth();
	}
	
	this.initParams = function()
	{
		var _this = this;
		
		this.bodyDimension = parseInt(
			this.dWindowDiv.select('.d3mocracy_avatar_body', '.d3mocracy_avatar_body_new')[0].style.height);
		this.headFootWidth = parseInt(this.dWindowDiv.select('.d3mocracy_avatar_head')[0].style.width);
		this.headFootWidthPadding = this.bodyDimension - this.headFootWidth;
		
		// Getting number of visible boxes
		this.numBoxes = 0;
		this.dWindowDiv.select('.d3mocracy_avatar_box').each(function(box)
		{
			// Count only non-hidden boxes
			if(box.style.display != 'none')
			{
				_this.numBoxes ++;
			}
		});
	}
	
	this.initHideButton = function()
	{
		var _this = this;
		
		this.hideButton = this.dWindowDiv.select('a.d3mocracy_hide_window')[0];
		this.hideButton.observe('click', function()
		{
			_this.dWindowDiv.hide();
		});
	}
	
	this.initAvatarBoxes = function()
	{
		var _this = this;
		
		this.dWindowDiv.select('.d3mocracy_avatar_box').each(function(item)
		{
			if(!item.hasClassName('d3mocracy_avatar_box_new'))
			{
				_this.createAvatarBox(item);
			}
		});
		
		this.avatarBoxNewDiv = this.dWindowDiv.select('.d3mocracy_avatar_box_new')[0];
		this.avatarBoxNewDiv.select('a.d3mocracy_new_avatar').each(function(item)
		{
			_this.middleAlignAddNewLink(item);
		});
		
		if(d3mocracyUserRole == 'exd3mocrat')
		{
			this.avatarBoxNewDiv.observe('click', function()
			{
				alert(d3mocracyActionsSuspended);
			});
		}
		else
		{
			this.avatarBoxNewDiv.select('a.d3mocracy_new_avatar')[0].observe('click', function()
			{
				window.open('d3mocracy.php?action=addavatar&ownerid=' + _this.uid, '', 'width=780,height=380');
			});
		}
	}
	
	this.middleAlignAddNewLink = function(addNewLink)
	{
		addNewLink.style.marginTop = (Math.floor((this.bodyDimension - 40) / 2)) + 'px';
	}
	
	this.resetAddNewBoxDimensions = function()
	{
		var divHead = this.avatarBoxNewDiv.select('.d3mocracy_avatar_head')[0];
		var divBody = this.avatarBoxNewDiv.select('.d3mocracy_avatar_body_new')[0];
		var divFoot = this.avatarBoxNewDiv.select('.d3mocracy_avatar_foot')[0];
		var addLink = this.avatarBoxNewDiv.select('a.d3mocracy_new_avatar')[0];
		
		divHead.style.width = this.headFootWidth + 'px';
		divFoot.style.width = this.headFootWidth + 'px';
		
		divBody.style.width = this.bodyDimension + 'px';
		divBody.style.height = this.bodyDimension + 'px';
		
		this.middleAlignAddNewLink(addLink);
	}
	
	this.resetDimensions = function(newBodyDimension)
	{
		this.bodyDimension = newBodyDimension;
		this.headFootWidth = newBodyDimension - this.headFootWidthPadding;
	}
	
	this.recalcWindowWidth = function()
	{
		if(this.numBoxes > 0)
		{
			var mp = this.numBoxes < d3mocracyWindowBoxWrap ? this.numBoxes : d3mocracyWindowBoxWrap;
			this.dWindowDiv.style.width = (mp * (this.bodyDimension + 10)) + 'px';
		}
	}
	
	this.repositionAvatarBox = function(dAvatarBox)
	{
		var _this = this;
		
		this.dWindowDiv.select('.d3mocracy_avatar_box').each(function(avatarBoxDivCurrent)
		{
			// Skip self
			if(dAvatarBox.dAvatarBoxDiv === avatarBoxDivCurrent)
			{
				return;
			}
			// Got Add New avatar box, no more items in loop, insert
			if(avatarBoxDivCurrent.hasClassName('d3mocracy_avatar_box_new'))
			{
				avatarBoxDivCurrent.insert({'before': dAvatarBox.dAvatarBoxDiv});
				throw $break;
			}
			// Getting wrapper object containing important params
			var dAvatarBoxCurrent = _this.getDAvatarBox(avatarBoxDivCurrent);
			
			// Ratings are equal. At this point, there is no need to iterate further
			if(dAvatarBox.avatarRating == dAvatarBoxCurrent.avatarRating)
			{
				// Older avatar has an advantage over the newer one
				if(dAvatarBox.avatarSubmitted < dAvatarBoxCurrent.avatarSubmitted)
				{
					avatarBoxDivCurrent.insert({'before': dAvatarBox.dAvatarBoxDiv});
				}
				// Newer avatar is inserted after the current one since ratings are equal
				else
				{
					avatarBoxDivCurrent.insert({'after': dAvatarBox.dAvatarBoxDiv});
				}
				throw $break;
			}
			// Voted rating is greater than current avatar's
			else if(dAvatarBox.avatarRating > dAvatarBoxCurrent.avatarRating)
			{
				avatarBoxDivCurrent.insert({'before': dAvatarBox.dAvatarBoxDiv});
				throw $break;
			}
		});
	}
	
	this.updateTooltips = function(avatarTooltips)
	{
		var _this = this;
		
		for(var i = 0; i < avatarTooltips.length; i++)
		{
			var avatarTooltip = avatarTooltips[i];
			var dAvatarBox = _this.dAvatarBoxes[avatarTooltip.avatarid];
			if(dAvatarBox)
			{
				if(avatarTooltip.submitted)
					dAvatarBox.divTipSubmitted.innerHTML = avatarTooltip.submitted;
				
				if(avatarTooltip.since)
					dAvatarBox.divTipSince.innerHTML = avatarTooltip.since;
				
				if(avatarTooltip.expires)
					dAvatarBox.divTipExpires.innerHTML = avatarTooltip.expires;
			}
		}
	}
	
	this.showAvatarBoxNew = function()
	{
		this.avatarBoxNewDiv.show();
		this.numBoxes ++;
	}
	
	this.hideAvatarBoxNew = function()
	{
		this.avatarBoxNewDiv.hide();
		this.numBoxes --;
	}
	
	this.removeAvatarBox = function(dAvatarBox)
	{
		delete this.dAvatarBoxes[dAvatarBox.avatarID];
		this.numBoxes --;
	}
	
	this.getNumAvatars = function()
	{
		return Object.keys(this.dAvatarBoxes).length;
	}
	
	this.construct(dWindowDiv, stAvatarLink, uid);
};

var DWindowListPrototype = function()
{
	this.construct = function()
	{
		this.dWindowList = {};
		this.loadingDiv = new Element('div', {'class': 'd3mocracy_loading'});
	}
	
	this.add = function(stAvatarContainer)
	{
		var _this = this;
		
		// User doesn't accept avatars, abort window init
		if(stAvatarContainer.select('input.d3mocracy_status')[0].value == 'off')
			return;
		
		var uid = stAvatarContainer.select('input.d3mocracy_uid')[0].value;
		var stAvatarLinks = stAvatarContainer.select('a');
		if(stAvatarLinks.length > 0)
		{
			// Set mouseover event, this is where window fetching occurs
			stAvatarLinks[0].observe('mouseover', function()
			{
				_this.initWindow(this, stAvatarContainer, uid);
			});
		}
	}
	
	this.initWindow = function(stAvatarLink, stAvatarContainer, uid)
	{
		var _this = this;
		
		this.hideWindows();
		
		// If window already exists
		if(this.dWindowList[uid])
		{
			if(stAvatarLink.previous() !== this.dWindowList[uid].dWindowDiv)
			{
				stAvatarLink.insert({'before': this.dWindowList[uid].dWindowDiv});
			}
			this.dWindowList[uid].dWindowDiv.show();
			
			// No need to do Ajax request
			return;
		}
		
		// Insert loading image
		stAvatarLink.insert({'after': this.loadingDiv});			
		
		// Window doesn't exist, fetch it
		new Ajax.Request('d3mocracy.php?action=window&ownerid='+uid+'&req='+Math.random(),
		{
			method: 'get',
			onSuccess: function(response) {_this.windowRequestSuccess(response, stAvatarLink, uid);},
			onFailure: function(response) {_this.windowRequestFailure();}
		});
	}
	
	this.windowRequestSuccess = function(response, stAvatarLink, uid)
	{
		this.removeLoadingDiv();
		
		// Alert if error
		if(response.responseJSON && response.responseJSON.error)
		{
			if(response.responseJSON.error.id == 'user_no_avatars')
			{
				stAvatarLink.stopObserving('mouseover');
			}
			else
			{
				alert(response.responseJSON.error.message);
			}
		}
		// Insert fetched window
		else if(response.responseText)
		{
			stAvatarLink.insert({'before': response.responseText});
			var dWindowDiv = stAvatarLink.previous('.d3mocracy_window');
			if(dWindowDiv)
			{
				this.dWindowList[uid] = new DWindowPrototype(dWindowDiv, stAvatarLink, uid);
			}
		}
	}
	
	this.windowRequestFailure = function()
	{
		// Remove loading and alert
		this.removeLoadingDiv();
		alert(d3mocracyLangRequestFailed);
	}
	
	this.hideWindows = function()
	{
		for(var w in this.dWindowList)
		{
			this.dWindowList[w].dWindowDiv.hide();
		}
	}
	
	this.removeLoadingDiv = function()
	{
		this.loadingDiv.remove();
	}
	
	this.construct();
};

var DEventStatic =
{
	suspendGet: function(observer, eventType)
	{
		var events = $(observer).getStorage().get('prototype_event_registry').get(eventType);
		$(observer).stopObserving(eventType);
		return events;
	},
	
	resume: function(observer, eventType, events)
	{
		events.each(function(event)
		{
			observer.observe(eventType, event.handler);
		});
	}
};

var DShowthreadStatic =
{
	updateAvatarCount: function(dWindow)
	{
		var avatarCounts = $$('.d3mocracy_avatar_count_uid_' + dWindow.uid);
		avatarCounts.each(function(item)
		{
			item.innerHTML = dWindow.getNumAvatars();
		});
	},
	
	resetActiveAvatar: function(dWindow)
	{
		var firstAvatarImg = dWindow.dWindowDiv.select('.d3mocracy_avatar_body img');
		var deleteStImages = false;
		
		// No images, showthread image will be deleted
		if(!firstAvatarImg.length)
		{
			deleteStImages = true;
		}
		// First image is same as showthread image; skip
		else if(firstAvatarImg.src == dWindow.stAvatarLink.select('img')[0].src)
		{
			return;
		}
		
		// Get user's showthread images
		var showThreadImgs = [];
		$$('span.d3mocracy_active_avatar').each(function(item)
		{
			if(item.select('input.d3mocracy_uid')[0].value == dWindow.uid)
			{
				var showThreadImg = item.select('img.showthread_avatar_img');
				if(showThreadImg.length > 0)
				{
					showThreadImgs.push(showThreadImg[0]);
				}
			}
		});
		showThreadImgs.each(function(showThreadImg)
		{
			// Set blank image
			if(deleteStImages)
			{
				showThreadImg.src = 'images/d3mocracy/humanity-user.png';
				showThreadImg.width = 66;
				showThreadImg.height = 78;
			}
			// Set new showthread image
			else
			{
				showThreadImg.src = firstAvatarImg[0].src;
				showThreadImg.width = parseInt(firstAvatarImg[0].style.width);
				showThreadImg.height = parseInt(firstAvatarImg[0].style.height);
			}
		});
	}
};

Event.observe(window, 'load', function()
{
	dWindowList = new DWindowListPrototype();
	$$('.d3mocracy_active_avatar').each(function(item)
	{
		dWindowList.add(item);
	});
});